"""
Manage the Mac pasteboard (aka "clipboard") supporting the
possibility of cutting and pasting rich text (HTML and RTF),
in addition to plain text.
"""

# Mac OS X: put export LC_CTYPE=en_US.utf-8 in .bash_profile to avoid anoying console UnicodeError


import Foundation
from AppKit import NSPasteboard

preferred_formats = ['rtf', 'html', 'text'] # in preference order

format2uti = {
    'html': 'public.html',
    'rtf':  'public.rtf',
    'text': 'public.utf8-plain-text'
}

# UTIs are Uniform Type Identfifiers (http://en.wikipedia.org/wiki/Uniform_Type_Identifier)
# ...even if they sound a lot like urinary tract infections

def paste(formats=None):
    """
    Get data from the Mac pasteboard (aka clipboard). Prefers a list of
    formats such as HTML and RTF, but takes plain text if that's what's available.
    That order of perferred formats has a reasonable default, but can be 
    specified on a per-call basis via the formats arg. If desired, raw UTIs
    can be given on formats as well.
    """
    formats = formats or preferred_formats
    pb = NSPasteboard.generalPasteboard()
    for f in formats:
        uti = format2uti.get(f, f)
        contents = pb.stringForType_(uti)
        if contents:
            return unicode(contents)
    return None

def available(dyn=False):
    """
    Returns list of UTIs (data types) of data available on the pasteboard. By
    default, excludes items whose type starts with 'dyn.', which is a dynamic
    type lookup scheme for 'unregistered' type codes--and beyond our current
    scope of operations.
    """
    pb = NSPasteboard.generalPasteboard()
    if dyn:
        return pb.pasteboardItems()[0].types()
    else:
        return [ t for t in pb.pasteboardItems()[0].types() if not t.startswith('dyn.') ] 

def pasteall(dyn=False):
    """
    Return a dict of all available data types, matched to the associated
    content. Good for curiosity, if not as useful in practice.
    """
    pb = NSPasteboard.generalPasteboard()
    return { t: pb.stringForType_(t) for t in available(dyn) }

def copy(**kwargs):
    """
    Put contents onto the Mac pasteboard (aka clipboard). Called with a kwargs
    style, such as copy(rtf=some_rtf, html=some_html). This lets a caller add
    multiple formats (representations) in parallel. As a backup, you can also
    provide a direct kwargs dict with actual UTIs.
    """
    pb = NSPasteboard.generalPasteboard()
    copy_utis = [ format2uti.get(f, f) for f in kwargs.keys() ]
    pb.declareTypes_owner_(copy_utis, None)

    for fmt, value in kwargs.iteritems():
        uti = format2uti.get(fmt, fmt)
        if isinstance(value, unicode):
            value = value.encode('utf-8')
        new_str = Foundation.NSString.stringWithString_(value).nsstring()
        new_data = new_str.dataUsingEncoding_(Foundation.NSUTF8StringEncoding)
        pb.setData_forType_(new_data, uti)

def clear():
    """
    Clears the existing contents of the pasteboard.
    """
    pb = NSPasteboard.generalPasteboard()
    pb.clearContents()
    
def uutf(v):
    """
    Coerce a value to unicode(), then get its utf8 encoding. 
    """
    return unicode(v).encode('utf-8')

def dict_print(d, heading=None, wrap=False, **kwargs):
    """
    Tidy printing of dictionary elements. Only here
    for demonstation and debugging.
    """
    import textwrap
    initial = kwargs.get('initial_indent', '')
    kwargs['subsequent_indent'] = kwargs.get('subsequent_indent', initial + '    ')
    
    if heading:
        print heading
    for key, value in d.iteritems():
        item_text = "{}: {}".format(uutf(key), uutf(value))
        if wrap:
            print '\n'.join(textwrap.wrap(item_text, **kwargs))
        else:
            print item_text
    
def test_richxerox():
    """
    Let's try it out!
    """
    print available()
    print paste()
    dict_print(pasteall(), 'ALL CONTENTS')
    copy(text="this is good!", html="this is <strong>good</strong>!")
    dict_print(pasteall(), 'AFTER COPY')
    
if __name__ == '__main__':
    test_richxerox()