"""
Manage the Mac pasteboard (aka "clipboard") supporting the
possibility of cutting and pasting rich text (HTML and RTF),
in addition to plain text.
"""

# Mac OS X: put export LC_CTYPE=en_US.utf-8 in .bash_profile to avoid anoying console UnicodeError


import Foundation
from AppKit import NSPasteboard

preferred_formats = ['rtf', 'html', 'text'] # in preference order

format2uti = {
    'html': 'public.html',
    'rtf':  'public.rtf',
    'text': 'public.utf8-plain-text'
}
uti2format =  dict((v,k) for k, v in format2uti.iteritems())

# UTIs are Uniform Type Identfifiers (http://en.wikipedia.org/wiki/Uniform_Type_Identifier)
# ...even if they sound a lot like urinary tract infections

def UTI(fmt):
    """
    Return the UTI for the given simple format name ('rtf', 'html', 'text').
    If the format name is unknown, returns it directly, assuming it's an
    explicit UTI.
    """
    return format2uti.get(fmt, fmt)

def paste(format='text'):
    """
    Return data of the given format ('rtf', 'html', 'text', or an explict UTI) from
    the Mac pasteboard (aka clipboard), if any, otherwise None.
    """
    pb = NSPasteboard.generalPasteboard()
    return unicode(pb.stringForType_(UTI(format))) or None

def available(neat=True, dyn=False):
    """
    Return list of formats available on the pasteboard. By default, provides a
    simple 'format' name ('rtf', 'html', or 'text') for preferred types, or UTIs
    for other types. By default, excluses items whose type starts with 'dyn.',
    which is a dynamic type lookup scheme for 'unregistered' type codes--beyond
    our current scope of operations.
    """
    items = NSPasteboard.generalPasteboard().pasteboardItems()
    if not items:
        return []
    types = items[0].types()
    if not dyn:
        types = [ t for t in types if not t.startswith('dyn.') ]
    if neat:
        return [ uti2format.get(u, u) for u in types ]
    else:
        return types
    
def pasteall(neat=True, dyn=False):
    """
    Return a dict of all available data types, matched to the associated
    content. Good for curiosity, if not as useful in practice. Arguments
    are as those for available().
    """
    pb = NSPasteboard.generalPasteboard()
    return { unicode(t): unicode(pb.stringForType_(UTI(t))) for t in available(neat, dyn) }

def copy(**kwargs):
    """
    Put contents onto the Mac pasteboard (aka clipboard). Called with a kwargs
    style, such as copy(rtf=some_rtf, html=some_html). This lets a caller add
    multiple formats (representations) in parallel. As a backup, you can also
    provide a direct kwargs dict with actual UTIs.
    """
    pb = NSPasteboard.generalPasteboard()
    pb.declareTypes_owner_([ UTI(f) for f in kwargs.keys() ], None)

    for fmt, value in kwargs.iteritems():
        if isinstance(value, unicode):
            value = value.encode('utf-8')
        new_str = Foundation.NSString.stringWithString_(value).nsstring()
        new_data = new_str.dataUsingEncoding_(Foundation.NSUTF8StringEncoding)
        pb.setData_forType_(new_data, UTI(fmt))

def clear():
    """
    Clears the existing contents of the pasteboard.
    """
    pb = NSPasteboard.generalPasteboard()
    pb.clearContents()


def dict_print(d, heading=None, wrap=False, **kwargs):
    """
    Tidy printing of dictionary elements. Only here
    for demonstation and debugging.
    """
    import textwrap
    initial = kwargs.get('initial_indent', '')
    kwargs['subsequent_indent'] = kwargs.get('subsequent_indent', initial + '    ')
    
    if heading:
        print heading
    for key, value in d.iteritems():
        item_text = "{}: {}".format(key, value)
        if wrap:
            print '\n'.join(textwrap.wrap(item_text, **kwargs))
        else:
            print item_text
    
def test_richxerox():
    """
    Let's try it out!
    """
    print available()
    print paste()
    dict_print(pasteall(), 'ALL CONTENTS')
    
    clear()
    dict_print(pasteall(), 'AFTER CLEAR')
    
    copy(text="this is good!", html="this is <strong>good</strong>!")
    dict_print(pasteall(), 'AFTER COPY')
    
if __name__ == '__main__':
    test_richxerox()