# coding=utf-8
"""
A helper script to find minimum mean rmsd structures in 2D-RMSd plot


Usage:
    rms2dfinder <rms2d output file>


Author:     Alisue <lambdalisue@hashnote.net>
URL:        http://hashnote.net/
License:    MIT License

(C) 2014, hashnote.net, Alisue
"""
__author__ = 'Alisue <lambdalisue@hashnote.net>'
import argparse
import itertools
import numpy as np
import matplotlib.pyplot as pl
from matplotlib.patches import Rectangle
from matplotlib.widgets import SpanSelector


def onselect(vmin, vmax):
    if vmin == vmax:
        # check if the user clicked on existing rectangle
        for r in reversed(onselect.rects):
            if r.get_x() <= vmin <= r.get_x() + r.get_width():
                # delete the rectangle
                r.text.remove()
                onselect.rects.remove(r)
                r.remove()
                onselect.ax.figure.canvas.draw()
                break
        return
    # find a frame index which have a smallest total RMSd
    subset = onselect.data[vmin:vmax, vmin:vmax]
    totals = np.sum(subset, axis=0)
    mmrms = np.min(totals) / len(totals)
    found = np.argmin(totals) + vmin

    t = onselect.ax.text(
        vmin + (vmax - vmin) / 2,
        vmin + (vmax - vmin) / 2,
        "RMSd: %.2f\nFrame: %d\nIn: %d-%d" % (mmrms, found, vmin, vmax),
        fontsize=10,
        va='center', ha='center',
        color='w',
    )
    r = Rectangle(
        (vmin, vmin),
        vmax-vmin, vmax-vmin,
        alpha=0.4,
        color=next(onselect.colors)
    )
    r.text = t
    onselect.rects.append(r)
    onselect.ax.add_patch(r)
    onselect.ax.figure.canvas.draw()


def plot_rms2d(fname):
    # load Amber 14 rms2d output (remove the first index line)
    data = np.loadtxt(fname)[:, 1:]

    # # single column graphics maximum wide = 3.33 in
    # figure_width = 3.33
    # figure_height = figure_width / 1.414    # silver
    # figure_height = figure_width / 1.618    # gold
    # pl.figure(figsize=(figure_width, figure_height))

    ax = pl.gca()
    im = ax.imshow(data, origin='lower')
    im.set_cmap('gray')

    rectprops = dict(
        color='yellow',
        alpha=0.1,
        fill=True,
    )
    onselect.ax = ax
    onselect.data = data
    onselect.rects = []
    onselect.colors = itertools.cycle(['c', 'm', 'y', 'r', 'g', 'b'])
    onselect.rectprops = rectprops
    onselect.ss = SpanSelector(
        ax, onselect, 'horizontal',
        useblit=True,
        rectprops=rectprops,
    )
    pl.show()


def main():
    parser = argparse.ArgumentParser(
        description=(
            "A helper utility to find minimum mean RMSd structures of "
            "clusters found in rms2d plot."
        ),
        epilog = "(C) 2014, hashnote.net, Alisue",
    )
    parser.add_argument(
        'infile', type=argparse.FileType('r'),
        help=(
            "A file generated by rms2d command in cpptraj of Amber."
        )
    )

    args = parser.parse_args()
    plot_rms2d(args.infile)


if __name__ == '__main__':
    main()
