import re

class RomanNumeralError(Exception): pass

class RomanNumeral(object):
    """
    The purpose of this class is to provide an easy way to perform
    conversion of decimal digits to Roman Numeral digits and vice-versa.
    Also, it's possible to perform most numeral operations such as
    addition, subtraction, multiplication and division. When performing
    such operations with an object from a different class, the object's
    class must implement __int__.

    Examples,
    >>> from romanpy import RomanNumeral
    >>> RomanNumeral(20)
    'XX'
    >>> rn = RomanNumeral('XIX')
    >>> rn
    'XIX'
    >>> str(rn)
    'XIX'
    >>> int(rn)
    19
    >>> rn < 100
    True
    >>> rn == 19
    True
    >>> rn > RomanNumeral('V')
    True
    >>> rn += 1
    >>> rn
    'XX'
    >>> int(rn)
    20
    """

    # Regular expression done by paxdiable on stackoverflow:
    # http://stackoverflow.com/questions/267399/
    # how-do-you-match-only-valid-roman-numerals-with-a-regular-expression
    __VALID_REGX = re.compile(
            '^M{0,3}(CM|CD|D?C{0,3})(XC|XL|L?X{0,3})(IX|IV|V?I{0,3})$')

    # Conversion table of shorthand digits to extended version.
    __SHORTHAND  = {'IV': 'IIII',
                    'IX': 'VIIII',
                    'XL': 'XXXX',
                    'XC': 'LXXXX',
                    'CD': 'CCCC',
                    'CM': 'DCCCC'}

    # All the possible Roman Numeral digits in ascending order.
    __DIGITS = ['I', 'IV', 'V', 'IX', 'X', 'XL', 'L',
                'XC', 'C', 'CD', 'D', 'CM', 'M']

    # Conversion table of Roman Numeral digits to decimal integers.
    __INT_CONVERSION = {'I' : 1,
                        'IV': 4,
                        'V' : 5,
                        'IX': 9,
                        'X' : 10,
                        'XL': 40,
                        'L' : 50,
                        'XC': 90,
                        'C' : 100,
                        'CD': 400,
                        'D' : 500,
                        'CM': 900,
                        'M' : 1000}

    # Minimum and maximum Roman Numeral values.
    __MIN_INT_VALUE = 0
    __MAX_INT_VALUE = 3999

    def __init__(self, val=''):
        """
        Val is used to create a new Roman Numeral from a specified
        value. Val can be a digit (int, float, double or complex) or
        a string of Roman Numeral digits, such as 'XIX'. The default
        value of val is an empty string which is treated as 0.
        """
        # Detect if it's an int, float, double or complex then
        # convert to Roman Numeral.
        if isinstance(val, (int, float, complex)):
            val = self.__toRomanNumeral(int(val))

        # Store value.
        self.__value = val
        # Verify value is a valid RomanNumeral.
        self.validate()

    # Value is implemented as a property in order to protect
    # it from external changes.
    def __getValue(self):
        return self.__value
    value = property(__getValue)

    def validate(self):
        """
        Verifies that the value is a valid RomanNumeral.
        """
        if not self.__isValid(self.__value):
            raise RomanNumeralError('Invalid value!')

    def __isValid(self, value):
        """
        Returns whether or not the given value holds a valid
        Roman numeral value.
        """
        if type(value) != str:
            return False
        # Empty values are ok.
        if len(value) == 0:
            return True
        return self.__VALID_REGX.match(value) is not None

    def split(self):
        """
        Splits the Roman Numeral into digits.
        """
        # Do a "dumb" split - assumes every character is a
        # Roman Numeral digit.
        digits = list(self.__value)
        # Combine shorthand values. ['I', 'X'] becomes ['IX',]
        for shorthand in self.__SHORTHAND.keys():
            while True:
                i = self.__pairIndex(list(shorthand), digits)
                if i == -1:
                    break
                # Replace pair with shorthand.
                digits = digits[:i] + [shorthand] + digits[i+2:]
        return digits

    ###########################################################################
    #
    # COMPARISON FUNCTIONS
    #
    ###########################################################################

    def __lt__(self, other):
        return int(self) < int(other)

    def __le__(self, other):
        return int(self) <= int(other)

    def __gt__(self, other):
        return int(self) > int(other)

    def __ge__(self, other):
        return int(self) >= int(other)

    def __eq__(self, other):
        return int(self) == int(other)

    def __ne__(self, other):
        return int(self) != int(other)

    def abc__cmp__(self, other):
        """
        Compares the value of two Roman Numeral objects.
        """
        return cmp(int(self), int(other))

    ###########################################################################
    #
    # NUMERICAL OPERATIONS
    #
    ###########################################################################

    def __add__(self, other):
        return RomanNumeral(int(self) + int(other))

    def __sub__(self, other):
        return RomanNumeral(int(self) - int(other))

    def __mul__(self, other):
        return RomanNumeral(int(self) * int(other))

    def __div__(self, other):
        return self.__floordiv__(other)

    def __truediv__(self, other):
        return self.__floordiv__(other)

    def __floordiv__(self, other):
        return RomanNumeral(int(self) // int(other))

    def __mod__(self, other):
        return RomanNumeral(int(self) % int(other))

    def __divmod__(self, other):
        q, r = divmod(int(self), int(other))
        return RomanNumeral(q), RomanNumeral(r)

    def __pow__(self, other, modulo=None):
        # Modulo can only be int or None.
        if modulo is not None:
            modulo = int(modulo)
        return RomanNumeral(pow(int(self), int(other), modulo))

    ###########################################################################
    #
    # CONVERSION METHODS
    #
    ###########################################################################

    def __int__(self):
        """
        Returns an integer representing the Roman Numeral. An empty
        Roman Numeral, '', is converted to 0.
        """
        return sum([self.__INT_CONVERSION[digit] for digit in self.split()])

    def __toRomanNumeral(self, int_val):
        """
        Returns a string representing the given int_val as a
        Roman Numeral.
        """
        # First check boundaries of given integer.
        if not self.__MIN_INT_VALUE <= int_val <= self.__MAX_INT_VALUE:
            raise RomanNumeralError('Numeral must be between %s and %s!' %
                                (self.__MIN_INT_VALUE, self.__MAX_INT_VALUE))
        rn_value = ''
        for rn in reversed(self.__DIGITS):
            amount, int_val = divmod(int_val, self.__INT_CONVERSION[rn])
            rn_value += amount * rn
        return rn_value

    def __str__(self):
        """
        Returns the string representation of the Roman Numeral.
        For example, XIX
        """
        return self.value

    def __repr__(self):
        """
        Returns the object representation of the Roman Numeral.
        For example, 'XIX'
        """
        return "'%s'" % self.value


    ###########################################################################
    #
    # HELPER METHODS
    #
    ###########################################################################

    def __pairIndex(self, pair, lst):
        """
        Returns the index of the first element of pair
        in lst. If not found, -1 is returned. A pair is considered
        found if, and only if, it occurs consecutively in list.
        For example, ['a','b'] is in ['a','b','c']
        but ['b','a'] is NOT in ['a','b','c']
        """
        # A pair must contain two elements.
        if len(pair) != 2:
            raise RomanNumeralError('A pair was not given, %s.' % str(pair))

        # Pair not in list because list is too small.
        if len(lst) < 2:
            return -1

        # Ensure both elements in pair are in lst.
        if pair[0] not in lst or pair[1] not in lst:
            return -1
        
        # Actually verify the elemetns in pair occur consecutively.
        for i in range(len(lst) - 1):
            if lst[i] == pair[0] and lst[i+1] == pair[1]:
                return i
        return -1

    def __cmpDigit(self, x, y):
        """
        Compares Roman Numeral's digits, such as IX and V.
        """
        if x == y:
            return 0
        for val in self.__DIGITS:
            if x == val:
                return -1
            elif y == val:
                return 1
        return 0

    ###########################################################################
    #
    # FUTURE METHODS
    #
    ###########################################################################

    #def sort(self):
    #    """
    #    Sort digits in descending order.
    #    """
    #    digits = self.split()
    #    digits.sort(self.__cmpDigit, reverse=True)
    #    self.__value = ''.join(digits)

    #def expand(self):
    #    """
    #    Expands shorthand notation.
    #    """
    #    for shorthand, expanded in self.__SHORTHAND.items():
    #        self.__value = self.__value.replace(shorthand, expanded)
                
    #def implode(self):
    #    """
    #    Applies shorthand notation.
    #    """
    #    for shorthand, expanded in self.__SHORTHAND.items():
    #        self.__value = self.__value.replace(expanded, shorthand)

