import unittest
from romanpy import RomanNumeral as Numeral
from romanpy import RomanNumeralError

class IsValid(unittest.TestCase):

    def setUp(self):
        self.n = Numeral()

    def testInvalidOneDigit(self):
        self.assertFalse(self.n._RomanNumeral__isValid('P'))

    def testValidOneDigit(self):
        self.assertTrue(self.n._RomanNumeral__isValid('X'))

    def testValidEmpty(self):
        self.assertTrue(self.n._RomanNumeral__isValid(''))

    def testValueNone(self):
        self.assertFalse(self.n._RomanNumeral__isValid(None))

class SizeChecking(unittest.TestCase):

    def testAddTooSmall(self):
        self.assertRaises(RomanNumeralError, Numeral, -1)

    def testSmallestNoException(self):
        Numeral(0)

    def testAddTooLarge(self):
        self.assertRaises(RomanNumeralError, Numeral, 4000)

    def testLargestNoException(self):
        Numeral(3999)

class PairIndex(unittest.TestCase):

    def setUp(self):
        self.n = Numeral()

    def testSmallPair(self):
        self.assertRaises(RomanNumeralError, self.n._RomanNumeral__pairIndex,
                          ['a'], ['a','b','c'])

    def testLargePair(self):
        self.assertRaises(RomanNumeralError, self.n._RomanNumeral__pairIndex,
                          ['a', 'b', 'c'], ['a','b','c','d'])

    def testSmallerList(self):
        self.assertRaises(RomanNumeralError, self.n._RomanNumeral__pairIndex,
                          ['a', 'b', 'c'], ['a','b'])

    def testNotFound(self):
        self.assertTrue(self.n._RomanNumeral__pairIndex(['c','d'], ['a','b','e'])
                        == -1)

    def testNotFoundEnd(self):
        self.assertTrue(self.n._RomanNumeral__pairIndex(['c','d'], ['a','b','c'])
                        == -1)

    def testFound(self):
        self.assertTrue(self.n._RomanNumeral__pairIndex(['b','c'], ['a','b','c'])
                        == 1)

class Split(unittest.TestCase):
    
    def testOneDigit(self):
        n = Numeral('I')
        self.assertTrue(n.split() == ['I'])

    def testSimple(self):
        n = Numeral('XVII')
        self.assertTrue(n.split() == ['X','V','I','I'])

    def testShorthand(self):
        n = Numeral('XCVII')
        self.assertTrue(n.split() == ['XC','V','I','I'])

#class sort(unittest.TestCase):
#
#    def testSimpleSort(self):
#        n = Numeral('VIX')
#        n.sort()
#        self.assertTrue(n.value == 'IXV')
#
#    def testWeirdSort(self):
#        n = Numeral('VIIX')
#        n.sort()
#        self.assertTrue(n.value == 'IXVI')

#class expand(unittest.TestCase):
#    
#    def testSimple(self):
#        n = Numeral('IV')
#        n.expand()
#        self.assertTrue(n.value == 'IIII')

#class implode(unittest.TestCase):
#    
#    def testSimple(self):
#        n = Numeral('IIII')
#        n.implode()
#        self.assertTrue(n.value == 'IV')
#
#    def testWeird(self):
#        n = Numeral('IIIII')
#        n.implode()
#        self.assertTrue(n.value == 'IVI')

class ConvertToInt(unittest.TestCase):

    def testOneSimpleDigit(self):
        self.assertTrue(int(Numeral('V')) == 5)

    def testOneCompoundDigit(self):
        self.assertTrue(int(Numeral('IV')) == 4)

    def testTwoSimpleDigits(self):
        self.assertTrue(int(Numeral('CMIV')) == 904)

    def testTwoCompoundDigits(self):
        self.assertTrue(int(Numeral('IV')) == 4)

class ConvertToRomanNumeral(unittest.TestCase):

    def testOneSimpleDigit(self):
        self.assertTrue(Numeral(1000).value == 'M')

    def testOneCompoundDigit(self):
        self.assertTrue(Numeral(900).value == 'CM')

    def testTwoSimpleDigits(self):
        self.assertTrue(Numeral(1010).value == 'MX')

    def testTwoCompoundDigits(self):
        self.assertTrue(Numeral(909).value == 'CMIX')

    def testTwoCompoundDigitsSame(self):
        self.assertTrue(Numeral(19).value == 'XIX')

    def testLargeComplicatedNumber(self):
        self.assertTrue(Numeral(3873).value == 'MMMDCCCLXXIII')

    def testRangeNoError(self):
        # This ensures that none of the int values cause
        # an invalid roman numeral (exception raised).
        # It also verifies that the numeral can be converted
        # back to an int successfully.
        for i in range(Numeral._RomanNumeral__MIN_INT_VALUE,
                       Numeral._RomanNumeral__MAX_INT_VALUE+1):
            self.assertTrue(int(Numeral(i)) == i)

class ConvertToString(unittest.TestCase):
    
    def testSimple(self):
        self.assertTrue(str(Numeral('XIX')) == 'XIX')

class ConvertToRepr(unittest.TestCase):
    
    def testSimple(self):
        self.assertTrue(repr(Numeral('XIX')) == "'XIX'")

class ObjectCmp(unittest.TestCase):

    def testEqualInt(self):
        self.assertTrue(Numeral('X') == 10)

    def testEqualRomanNumeral(self):
        self.assertTrue(Numeral('X') == Numeral('X'))

    def testEqualFloat(self):
        self.assertTrue(Numeral('X') == 10.0)

    def testGreaterThanInt(self):
        self.assertTrue(Numeral('X') > 9)

    def testGreaterThanRomanNumeral(self):
        self.assertTrue(Numeral('X') > Numeral('IX'))

    def testGreaterThanFloat(self):
        self.assertTrue(Numeral('X') > 9.999)

    def testLessThanInt(self):
        self.assertTrue(Numeral('X') < 19)

    def testLessThanRomanNumeral(self):
        self.assertTrue(Numeral('X') < Numeral('XIX'))

    def testLessThanFloat(self):
        self.assertTrue(Numeral('X') < 19.999)

class Addition(unittest.TestCase):

    def testAddInt(self):
        self.assertTrue((Numeral('X') + 9).value == 'XIX')

    def testAddRomanNumeral(self):
        self.assertTrue((Numeral('X') + Numeral('IX')).value == 'XIX')

    def testAddFloat(self):
        self.assertTrue((Numeral('X') + 9.692).value == 'XIX')

class Subtraction(unittest.TestCase):

    def testSubInt(self):
        self.assertTrue((Numeral('X') - 8).value == 'II')

    def testSubRomanNumeral(self):
        self.assertTrue((Numeral('X') - Numeral('VIII')).value == 'II')

    def testSubFloat(self):
        self.assertTrue((Numeral('X') - 8.87).value == 'II')

class Multiplication(unittest.TestCase):

    def testSubInt(self):
        self.assertTrue((Numeral('X') * 8).value == 'LXXX')

    def testSubRomanNumeral(self):
        self.assertTrue((Numeral('X') * Numeral('VIII')).value == 'LXXX')

    def testSubFloat(self):
        self.assertTrue((Numeral('X') * 8.87).value == 'LXXX')

class Division(unittest.TestCase):

    def testDivInt(self):
        self.assertTrue((Numeral('X') / 2).value == 'V')

    def testDivRomanNumeral(self):
        self.assertTrue((Numeral('X') / Numeral('II')).value == 'V')

    def testSubFloat(self):
        self.assertTrue((Numeral('X') / 2.87).value == 'V')

    def testFloorDivInt(self):
        self.assertTrue((Numeral('X') // 2).value == 'V')

    def testModInt(self):
        self.assertTrue((Numeral('X') % 3).value == 'I')

    def testDivModInt(self):
        q, r = divmod(Numeral('X'), 3)
        self.assertTrue(q.value == 'III')
        self.assertTrue(r.value == 'I')
 
class Power(unittest.TestCase):

    def testPowInt(self):
        self.assertTrue(pow(Numeral('II'), 3).value == 'VIII')

    def testPowRomanNumeral(self):
        self.assertTrue(pow(Numeral('II'), Numeral('III')).value == 'VIII')

    def testPowFloat(self):
        self.assertTrue(pow(Numeral('II'), 3.893).value == 'VIII')

    def testPowIntModulo(self):
        self.assertTrue(pow(Numeral('II'), 3, 5).value == 'III')

    def testPowRomanNumeralModulo(self):
        self.assertTrue(
                pow(Numeral('II'), Numeral('III'), Numeral('V')).value == 'III')

    def testPowModulo(self):
        self.assertTrue(pow(Numeral('II'), 3.893, 5.88).value == 'III')

if __name__ == '__main__':
    unittest.main()
