import os
import sys
import re
import argparse
import time
from subprocess import Popen, PIPE

import logging
logging.basicConfig(format='%(levelname)s:%(name)s:line %(lineno)s: %(message)s')
log = logging.getLogger(__name__)

try:
    from rsbactools import rsbac, RSBAC_PROC_INFO_DIR, class_for_name
except ImportError as error:
    print(error)
    sys.exit(-1)


__version__ = "0.0.2"


class Date(object):

    def __init__(self):
        import datetime
        self.date = datetime.datetime.now()
        
    def get_day(self, as_string=False):
        if as_string:
            return str(self.date.day)
        else:
            return self.date.day

    def get_hour(self, as_string=False):
        if as_string:
            return str(self.date.hour)
        else:
            return self.date.hour

    def get_month(self, as_string=False):
        if as_string:
            return str(self.date.month)
        else:
            return self.date.month

    def get_year(self, as_string=False):
        if as_string:
            return str(self.date.year)
        else:
            return self.date.year
    
    def get_timestamp(self, as_string=False, cut=False):
        timestamp = self.date.timestamp()
        if cut:
            timestamp = "%0.f" % timestamp
        if as_string:
            return str(timestamp)
        else:
            return timestamp


class DataStructure(object):
    
    def __init__(self, verbose=False):
        # for each entry a class exists and return with get_cmd() a list
        self.modules_to_backup = ["general", "log", "user", "net"]
        date = Date()
        self.timestamp = date.get_timestamp(as_string=True, cut=True)
        self.info = {
            "backup-dir": "backup",
            "year": date.get_year(True),
            "month": date.get_month(True),
            "day": date.get_day(True),
            "files": {},
        }

    
    def set_backup_dir(self, backup_dir):
        """update backupdir"""
        if backup_dir: 
            self.info["backup-dir"] = backup_dir
        else:
            self.info["backup-dir"] = os.path.join(os.getcwd(),
                self.info["backup-dir"])

    def set_modules_to_backup(self, modules):
        """create logfile names and update info.files"""
        self.modules_to_backup.extend(modules)
        for i in self.modules_to_backup:
            self.info["files"][i] = "_".join([i.lower(), self.timestamp])

    def get_info(self):
        return self.info

    def get_timestamp(self):
        return self.timestamp

    def create_dir_structure(self):
        try:
            if not os.path.isdir(self.info["backup-dir"]):
                log.info("create dir: %s" % self.info["backup-dir"])
                os.mkdir(self.info["backup-dir"])
            year_dir = os.path.join(self.info["backup-dir"], self.info["year"])
            if not os.path.isdir(year_dir):
                log.info("create dir: %s" % year_dir)
                os.mkdir(year_dir)
            month_dir = os.path.join(year_dir, self.info["month"])
            if not os.path.isdir(month_dir):
                log.info("create dir: %s" % month_dir)
                os.mkdir(month_dir)
            day_dir = os.path.join(month_dir, self.info["day"])
            if not os.path.isdir(day_dir):
                os.mkdir(day_dir)
            self.info["today"] = day_dir
            return True
        except OSError as error:
            log.error(error)


class BackupLog(object):
    
    def __init__(self):
        self.cmd = ["switch_adf_log", "-b"]
 
    def get_cmd(self):
        return self.cmd

class BackupNet(object):

    def __init__(self):
        self.cmd = [["net_temp", "-a", "-b"], 
            ["attr_back_net", "-a", "NETDEV"], 
            ["attr_back_net", "-a", "NETTEMP"],
       ] 

    def get_cmd(self):
        return tuple(self.cmd)

class BackupGeneral(object):

    def __init__(self):
        self.cmd = []
        exclude = ["JAIL"]
        for module, status in rsbac.Rsbac().get_modules()["Module"].items():
            if module in exclude:
                continue
            if status == "on":
                self.cmd.append(["attr_back_fd", "-r", "-M", module, "/"])
        self.cmd.append(["attr_back_dev", "b"])

    def get_cmd(self):
        return tuple(self.cmd)
    
class BackupAuth(object):

    def __init__(self):
        self.cmd = ["auth_back_cap", "-r", "/"]

    def get_cmd(self):
        return (self.cmd)

class BackupUm(object):

    def __init__(self):
        self.cmd = [["rsbac_groupshow", "-S", "all", "-b", "-p", "-a"],
            ["rsbac_usershow", "-S", "all", "-b", "-p", "-a"]
        ]
    
    def get_cmd(self):
        return tuple(self.cmd)

class BackupRc(object):

    def __init__(self):
        self.cmd = ["rc_get_item", "backup"]

    def get_cmd(self):
        return tuple(self.cmd)

class BackupUser(object):

    def __init__(self):
        self.cmd = []
        exclude = ["JAIL"]
        for module, status in rsbac.Rsbac().get_modules()["Module"].items():
            if module in exclude:
                continue
            if status == "on":
                self.cmd.append(["attr_back_user", "-a", "-M", module])
            
    def get_cmd(self):
        return tuple(self.cmd)


class Backup(object):
    """Backup RSABAC attribute modules based."""

    def __init__(self):
        self.rsbac = rsbac.Rsbac()
        self.args = {}
        self.data_structure = DataStructure()
        self.data_structure.set_modules_to_backup(self.get_modules_to_backup())
   
    def set_args(self, args):
        self.args = args

    def set_log_level(self, log_level):
        log.setLevel(log_level)

    def get_log_level(self):
        return log.getEffectiveLevel()

    def is_um_module_active(self):
        """Workaround to get a status if the UM enabled.
        When enabled file should exists.
        """
        if os.path.exists(os.path.join(RSBAC_PROC_INFO_DIR, "stats_um")):
            return True
    
    def get_modules_to_backup(self):
        """Return a list of active modules."""
        modules = []
        for module, status in self.rsbac.get_modules()["Module"].items():
            if status == "on":
                modules.append(module.lower())
        if self.is_um_module_active():
            modules.append("um")
        return modules

    def get_cmd_modules(self):
        return self.data_structure.get_info()["files"]

    def execute(self, cmd, backup_file, error_file, cmd_file):
        with open(backup_file, "w") as bf, open(error_file, "w") as ef, \
                open(cmd_file, "w") as cf:
            if type(cmd) is tuple:
                for i in cmd:
                    cf.write(" ".join([str(x) for x in i]) + "\n")
                    process = Popen(i, stdin=PIPE, stdout=bf, stderr=ef)
                    process.wait()
                    bf.flush()
            else:
                cf.write(" ".join([str(x) for x in cmd]) + "\n")
                process = Popen(cmd, stdin=PIPE, stdout=bf, stderr=ef)
                process.wait()
                bf.flush()
 
    def run(self):
        if self.args["directory"]:
            self.data_structure.set_backup_dir(self.args["directory"])
        self.data_structure.create_dir_structure()
        info = self.data_structure.get_info()

        for module, module_file in info["files"].items():
            if not self.args[module]:
                continue
            try:
                backup_file = os.path.join(info["today"], 
                    "_".join([module, self.data_structure.get_timestamp()]))
                error_file = ".".join([backup_file, "error"])
                cmd_file = ".".join([backup_file, "command"])
                backup_file =  ".".join([backup_file, "backup"])
               
                class_name = "".join(["Backup", module.capitalize()])
                cmd = class_for_name("rsbactools.backup", class_name)().get_cmd()
                try:
                    log.info("starting backup: %s" % module)
                    self.execute(cmd, backup_file, error_file, cmd_file)
                    log.info("done ...")
                except Exception as error:
                    log.error(error)
            except AttributeError as error:
                log.info("Not implemented: %s()" % class_name)

