from __future__ import division

from tabulate import tabulate


class SimpleTabulator():
    def __init__(self, results, title):
        """Accepts an iterable of (function, timingreport) pairs and a title
        for the comparison table."""
        self._results = results
        self._title = title

    def render_table(self):
        """Generate the table for this object's results as a string."""
        if not self._results:
            return ''

        # find the string for the units and the divisor to convert from usecs
        smallest = min(result.best for function, result in self._results)
        units, unit_divisor = self.units_and_divisor(smallest)
        # convert each result to its display value
        normalized_results = tuple(
            (func, result._replace(best=result.best / unit_divisor))
            for func, result in self._results)

        table = tuple(self.get_row(func, result)
                      for func, result in normalized_results)
        return tabulate(table, tablefmt='simple', floatfmt='.2f',
                        headers=self.header(self._title, units))

    def get_row(self, func, result):
        return (func.__name__,
                result.best,
                result.number,  # number of loops / repeat
                result.repeat)

    def format_title(self, title):
        max_len = 23
        if len(title) <= max_len:
            return title
        ellipsis = '...'
        return title[:max_len - len(ellipsis)] + ellipsis

    def header(self, title, units):
        """Generate the header for the table generated by this object as a
        string."""
        return (self.format_title(title), units, 'loops', 'best of')

    def units_and_divisor(self, smallest):
        """Get the unit name and a number that converts to those units from
        usecs"""
        for name, div in (('sec', 1000000), ('msec', 1000)):
            if smallest >= div:
                return name, div
        return 'usec', 1


def generate_table(results, title):
    return SimpleTabulator(results, title).render_table()
