from copy import copy
from functools import partial
from contextlib import contextmanager


class Dialog:
    """Dialog representation.

    Parameters
    ----------
    kwargs: dict
        Key=value pairs of dialog parameters.
    """

    # Public

    attempts = 1
    """How many attempts user have to match one of the available options.
    """
    default = None
    """Default value if user will not input anything.
    """
    fallback = None
    """Fallback value.
    """
    notice = None
    """Write when user fails to match one of the available option.
    """
    reader = staticmethod(input)
    """Base reader function. For example you can use getpass.getpass.
    """
    options = None
    """List of available options.
    """
    writer = staticmethod(partial(print, end=''))
    """Base writer function. For example you can use pprint.pprint.
    """

    def __init__(self, **kwargs):
        vars(self).update(kwargs)

    def __getitem__(self, key):
        value = getattr(self, key)
        return value

    def __getattribute__(self, name):
        value = super().__getattribute__(name)
        if isinstance(value, str):
            value = value.format_map(self)
        return value

    def listen(self, **kwargs):
        with self.__patch(**kwargs):
            return self.reader()

    def message(self, message, **kwargs):
        with self.__patch(**kwargs):
            message = message.format_map(self)
            return self.writer(message)

    def question(self, question='', **kwargs):
        with self.__patch(**kwargs):
            question = question.format_map(self)
            for _ in range(self.attempts):
                self.writer(question)
                answer = self.reader()
                if self.default is not None:
                    if not answer:
                        answer = self.default
                if self.options is not None:
                    if answer not in self.options:
                        if self.notice is not None:
                            self.writer(self.notice)
                        continue
                return answer
            return self.fallback

    @property
    def hint(self):
        elements = []
        options = copy(self.options)
        if self.options is None:
            options = ['*']
            if self.default is not None:
                options = [self.default] + options
        for option in options:
            if option == self.default:
                option = '[' + option + ']'
            elements.append(option)
        hint = '/'.join(elements)
        return hint

    # Private

    @contextmanager
    def __patch(self, **kwargs):
        attrs = vars(self)
        buffer = copy(attrs)
        attrs.update(**kwargs)
        try:
            yield
        finally:
            attrs.clear()
            attrs.update(**buffer)
