import os
from sugarbowl import Function
from jinja2 import FileSystemLoader
from ..environment import Environment


class render_file(Function):
    """Render a file using context.

    Parameters
    ----------
    filepath: str
        Path to template file.
    context: dict/obj
        Rendering context.
    params: dict
        Parameters to pass to jinja2's Environment.

    Returns
    -------
    str
        Rendered file.

    Notes
    -----
    There is two type of behaviour depending on params's loader argument:

    - If loader've been passed value of filepath will be a template name.
    - If not FileSystemLoader(os.path.dirname(filepath)) will be a loader,
      os.path.basename(filepath) will be a template name.
    """

    # Public

    def __init__(self, filepath, *, context=None, **params):
        self.__filepath = filepath
        self.__context = context
        self.__params = params

    def __call__(self):
        name = self.__filepath
        loader = self.__params.pop('loader', None)
        if loader is None:
            name = os.path.basename(self.__filepath)
            dirname = os.path.dirname(self.__filepath)
            loader = FileSystemLoader(dirname)
        environment = Environment(loader=loader, **self.__params)
        template = environment.get_template(name)
        result = template.render(self.__context)
        return result
