"""Interpolating string formatter. """

import string, inspect, sys, itertools
from chainstuf import chainstuf  
from options import Options, OptionsContext
import six

# Define convenience functions

def is_string(v):
    """
    Is the value v a string? Useful especially in making a test that works on
    both Python 2.x and Python 3.x
    """
    return isinstance(v, six.string_types)

def flatten(*args):
    """
    Like itertools.chain(), but will pretend that single scalar values are singleton lists.
    Convenient for iterating over values whether they're lists or singletons.
    """
    flattened = [ x if isinstance(x, (list, tuple)) else [x] for x in args ]
    return itertools.chain(*flattened)

    # would use ``hasattr(x, '__iter__')`` rather than ``isinstance(x, (list, tuple))``,
    # but other objects like file have ``__iter__``, which screws things up

# Workhorse functions

sformatter = string.Formatter()  # piggyback Python's format() template parser

def _sprintf(arg, caller, override=None):
    """
    Format the template string (arg) with the values seen in the context of the caller.
    If override is defined, it is a dictionary like object mapping additional
    values atop those of the local context.
    """

    def seval(s):
        """
        Evaluate the string s in the caller's context. Return its value.
        """
        try:
            localvars = caller.f_locals if override is None \
                                        else chainstuf(override, caller.f_locals)
            return eval(s, caller.f_globals, localvars)
        except SyntaxError:
            raise SyntaxError("syntax error when formatting '{}'".format(s))
        except StandardError as se:
            raise se

    if is_string(arg):
        parts = []
        for (literal_text, field_name, format_spec, conversion) in sformatter.parse(arg):
            parts.append(literal_text)
            if field_name is not None:
                format_str = six.u("{") + ("!" + conversion if conversion else "") + \
                                          (":" + format_spec if format_spec else "") + "}"
                field_value = seval(field_name)
                formatted = format_str.format(field_value)
                parts.append(formatted)
        return ''.join(parts)
    else:
        return str(seval(str(arg)))


class Say(object):
    
    """
    Say encapsulates printing functions. Instances are configurable, and callable.
    """
    
    options = Options(  
        encoding='utf-8',   # character encoding for output
        encoded=None,       # character encoding to return
        file=sys.stdout,    # where is output headed? a write() able object, or str
        sep=' ',            # separate args with this (Python print function compatible)
        end='\n',           # end output with this (Python print function compatible)
        silent=False,       # do the formatting and return the result, but don't write the output
        _callframe=None,    # frome from which the caller was calling
    )
  
    def __init__(self, **kwargs):
        """
        Make a say object with the given options.
        """
        self.options = Say.options.push(kwargs)
    
    def hr(self, sep=six.u('\u2500'), width=40, vsep=0, **kwargs):
        """
        Print a horizontal line. Like the HTML hr tag. Optionally
        specify the width, character repeated to make the line, and vertical separation.
        
        Good options for the separator may be '-', '=', or parts of the Unicode 
        box drawing character set. http://en.wikipedia.org/wiki/Box-drawing_character
        """
        opts = self.options.push(kwargs)

        self.blank_lines(vsep, **opts)
        self._output([sep * width], opts)
        self.blank_lines(vsep, **opts)

    def title(self, name, sep=six.u('\u2500'), width=15, vsep=0, **kwargs):
        """
        Print a horizontal line with an embedded title. 
        """
        opts = self.options.push(kwargs)

        self.blank_lines(vsep, **opts)
        line = sep * width
        self._output([ ' '.join([line, name, line]) ], opts)
        self.blank_lines(vsep, **opts)
    
    def blank_lines(self, n, **kwargs):
        """
        Output N blank lines ("vertical separation")
        """
        if n > 0:
            opts = self.options.push(kwargs)
            self._write("\n" * n, opts)

    def set(self, **kwargs):
        """
        Permanently change the reciver's settings to those defined in the kwargs.
        An update-like function.
        """
        self.options.set(**kwargs)
    
    def settings(self, **kwargs):
        """
        Open a context manager for a `with` statement. Temporarily change settings
        for the duration of the with.
        """
        return SayContext(self, kwargs)
    
    def clone(self, **kwargs):
        """
        Create a new Say instance whose options are chained to this instance's
        options (and thence to Say.options). bkwargs become the cloned instance's
        overlay options.
        """
        cloned = Say()
        cloned.options = self.options.push(kwargs)
        return cloned
    
    def setfiles(self, *files):
        """
        Set the list of output files. ``files`` is a list. For each item, if
        it's a real file like ``sys.stdout``, use it. If it's a string, assume
        it's a filename and open it for writing. 
        """
        def opened(f):
            """
            If f is a string, consider it a file name and return it, ready for writing.
            Else, assume it's an open file. Just return it.
            """
            return open(f, "w") if is_string(f) else f

        self.options.file = [ opened(f) for f in flatten(*files) ]
        return self

    def __call__(self, *args, **kwargs): 
        
        opts = self.options.push(kwargs)  
        opts._callframe = inspect.currentframe().f_back
        
        formatted = [ _sprintf(arg, opts._callframe) if is_string(arg) else str(arg)
                      for arg in args ]
        return self._output_str(opts.sep.join(formatted), opts)
        
    def _output_str(self, outstr, opts):
        """
        Do the actual formatting and outputting work. 
        """
        
        def encoded(s, encoding):
            """
            Encode the strings if they're encoding to be done.
            """
            return s.encode(encoding) if encoding else s

        # prepare and emit output
        if opts.end is not None:
            outstr += opts.end
        if not opts.silent:
            self._write(encoded(outstr, opts.encoding), opts)
            
        # prepare and return return value
        retencoding = opts.encoding if opts.encoded is True else opts.encoded
        retencoded = encoded(outstr, retencoding)
        rettrimmed = retencoded[:-1] if retencoded.endswith('\n') else retencoded
        return rettrimmed
        
    def _output(self, lines, opts):
        """
        Alternate output function that accepts list of lines, not composed strings.
        
        More important and logical in the future. Stay tuned.
        """
        self._output_str("\n".join(lines), opts)
    
    def _write(self, s, opts):
        """
        Write s to all associated file objects. 
        """
        for f in flatten(opts.file):
            f.write(s)


class SayContext(OptionsContext):
    """
    Context helper to support Python's with statement.  Generally called
    from ``with say.settings(...):``
    """
    pass
    
say = Say()
fmt = say.clone(silent=True).setfiles()
