

import six, os
from say import Say, fmt, stdout

def setup():
    global say
    say = Say(retvalue=True)

def test_basic(param = 'Yo'):
    setup()
    
    greeting = "hello"
    assert say("{greeting}, world!") == "{0}, world!".format(greeting)
    assert say("'{greeting}' has {len(greeting)} characters") == "'{0}' has {1} characters".format(greeting, len(greeting))
    assert say("{param}") == "{0}".format(param)
    assert say("{greeting:<10}") == "{0:<10}".format(greeting)
    assert say("{greeting:>20}") == "{0:>20}".format(greeting)
    
    status = "OK"
    assert say("This is {status}", silent=True) == "This is OK"
    
def test_hr_and_title():
    setup()
    
    say.title('say testing follows')
    say.hr(vsep=1)

    # http://en.wikipedia.org/wiki/Box-drawing_character
    say.hr(sep=six.u('\u2504'))
    say.hr(sep=six.u('\u2509'))
    say.hr(sep=six.u('\u2550'))
    say.hr(sep=six.u('\u2591'))
    say.hr(sep=six.u('\u25EF '))
    say.hr(sep=six.u('\u25C9 '))
    
def test_localvar():
    setup()

    m = "MIGHTY"
    assert say(m) == "MIGHTY"
    assert say("{m}") == "MIGHTY"
    globalvar = "tasty"    # override global var
    assert say(globalvar) == "tasty"
    assert say("{globalvar}") == "tasty"
    
def test_globalvar():
    setup()

    assert say("{globalvar}") == str(globalvar)
    
def test_unicode():
    setup()
    
    u = six.u('This\u2014is Unicode!')
    assert say(u) == u
    assert say("Hey! {u}") == six.u('Hey! This\u2014is Unicode!')
    too = "too"
    assert say(six.u("Unicode templates {too}")) == six.u("Unicode templates too")
    
def test_format_string():
    setup()

    x = 33.123456789
    assert say("{x} is floating point!") == '33.123456789 is floating point!'
    assert say("{x:0.2f} is shorter") == '33.12 is shorter'
    assert say("{x:8.3f} is in the middle") == '  33.123 is in the middle'
    
def test_fmt():
    setup()

    v = 1212 
    
    assert say("++{v}") == fmt("++{v}")
    # should print only once
    
def test_encoded_encoding():
    setup()
    say.set(encoding='base64', encoded=True)
    assert say('I am a truck!')  == "SSBhbSBhIHRydWNrIQo="
    assert say('I am a truck!', encoding='rot-13') == 'V nz n gehpx!'
    assert say('V nz n gehpx!', encoding='rot-13') == 'I am a truck!'
    
def test_files(tmpdir):
    say = Say()
    
    tmpfile = tmpdir.join('test.txt')
    say.setfiles([stdout, tmpfile])
    say("Yowza!")
    
    assert tmpfile.read() == "Yowza!" + "\n" 

def test_example_1():
    setup()
    
    x = 12
    nums = list(range(4))
        
    assert say("There are {x} things.") == "There are 12 things."
    assert say("Nums has {len(nums)} items: {nums}") == "Nums has 4 items: [0, 1, 2, 3]"
     
def test_example_2():
    setup()

    errors = [ {'name': 'I/O Error', 'timestamp': 23489273 },
               {'name': 'Compute Error', 'timestamp': 349734289 }]
    
    say.title('Errors')
    for i,e in enumerate(errors, start=1):
        say("{i:3}: {e['name'].upper()}")
        
def test_say_silent():
    setup()
    r1 = say("this is output")
    r2 = say("this is output", silent=True)
    assert r1 == r2
  
globalvar = 99
