# Copyright 2014, Scalyr, Inc.
#
# Contains the base class for all monitor plugins used by the Scalyr agent.
# This class should be used by developers creating their own monitor plugins.
#
# To see how to write your own Scalyr monitor plugin, please see:
# http://www.scalyr.com/FILL_THIS_IN  (TODO: Fill in after alpha once we have
# a permanent URL for the documentation.)

from threading import Lock

import scalyr_agent.scalyr_logging as scalyr_logging

from scalyr_agent.util import StoppableThread

log = scalyr_logging.getLogger(__name__)


class ScalyrMonitor(StoppableThread):
    """The base class for all monitors used by the agent.

    An instance of a monitor will be created for every reference to this module in the "monitors"
    section in the agent.json configuration file.  The monitor will be
    executed in its own thread and will be expected to send all output to
    provided Loggers.  Do not used stdout, stderr.

    Public attributes:  (must be updated in derived constructor)
        log_config:  A dict containing the log configuration object that
            should be used to configure the agent to copy the log generated
            by this module.  It has the same format as the entries in the
            "logs" section in agent.json.  In particular, the module
            can use this to specify the path of the log file where all emitted metric values
            from this monitor will be sent (using self._logger), set attributes to
            associate with all log lines generated by the module, specify a parser
            for the log, as well as set sampling rules.

            Note, if the "path" field in "log_config" is not absolute, it will be resolved relative to the
            directory specified by the "agent_log_path" option in the configuration file.
        disabled:  A boolean indicating if this module instance should be
            run.
    """

    def __init__(self, monitor_config, logger, sample_interval_secs=30):
        """Constructs an instance of the monitor.

        It is optional for derived classes to override this method.  The can instead
        override _initialize which is invoked during initialization.
        TODO:  Determine which approach is preferred by developers and recommend that.

        If a derived class overrides __init__, they must invoke this method in the override method.

        This method will set default values for
        all public attributes (log_config, disabled, etc).  These
        may be overwritten by the derived class.

        The derived classes must raise an Exception (or something derived from Exception)
        in this method if the provided configuration is invalid or if there is any other
        error known at this time preventing the module from running.

        Arguments:
            module_config:  A dict containing the configuration information
                for this module instance from the configuration file.  The
                only valid values are strings, ints, longs, floats, and booleans.
            logger:  The logger to use for output.
            sample_interval_secs:  The interval in seconds to wait between gathering
                samples.
        """
        # The MonitorConfig object created from the config for this monitor instance.
        self._config = MonitorConfig(monitor_config)
        # The logger instance that this monitor should use to report all information and metric values.
        self._logger = logger
        self.monitor_name = monitor_config['module']
        log_path = self.monitor_name.split('.')[-1] + '.log'
        self.disabled = False
        # TODO: For now, just leverage the logic in the loggers for naming this monitor.  However,
        # we should have it be more dynamic where the monitor can override it.
        if logger.component.find('monitor:') == 0:
            self.monitor_name = logger.component[8:]
        else:
            self.monitor_name = logger.component
        self.log_config = {
            "path": log_path,
        }
        # This lock protects all variables that can be access by other threads, reported_lines,
        # emitted_lines, and errors.  It does not protect _run_state since that already has its own lock.
        self.__lock = Lock()
        self.__reported_lines = 0
        self.__errors = 0

        self._sample_interval_secs = sample_interval_secs
        self.__metric_log_open = False
        self._initialize()

        StoppableThread.__init__(self, name='metric thread')

    def _initialize(self):
        """Can be overridden by derived classes to perform initialization functions before the monitor is run.

        This is meant to allow derived monitors to perform some initialization and configuration validation
        without having to override the __init__ method (and be responsible for passing all of the arguments
        to the super class).

        The derived classes must raise an Exception (or something derived from Exception)
        in this method if the provided configuration is invalid or if there is any other
        error known at this time preventing the module from running.
        """
        pass

    def reported_lines(self):
        """Returns the number of metric lines emitted to the metric log for this monitor.

        This is calculated by counting how many times the logger instance on this monitor's report_values
        method was invoked and all the times any logger has logged anything with metric_log_for_monitor set
        to this monitor.
        """
        self.__lock.acquire()
        result = self.__reported_lines
        self.__lock.release()
        return result

    def errors(self):
        """Returns the number of errors experienced by the monitor as it is running.

        For monitors just implementing the gather_sample method, this will be the number of times
        that invocation raised an exception.  If a monitor overrides the run method, then it is up to
        them to increment the errors as appropriate using increment_counter.
        """
        self.__lock.acquire()
        result = self.__errors
        self.__lock.release()
        return result

    def increment_counter(self, reported_lines=0, errors=0):
        """Increment some of the counters pertaining to the performance of this monitor.
        """
        self.__lock.acquire()
        self.__reported_lines += reported_lines
        self.__errors += errors
        self.__lock.release()

    def run(self):
        """Begins executing the monitor, writing metric output to logger.

        Implements the business logic for this monitor.  This method will
        be invoked by the agent on its own thread.  This method should
        only return if the monitor instance should no longer be executed or
        if the agent is shutting down.

        The default implementation of this method will invoke the
        "gather_sample" once every sample_period time, emitting the returned
        dict to logger.  Derived classes may override this method if they
        wish.

        This method should use "self._logger" to report any output.  It should use
        "self._logger.emit_value" to report any metric values generated by the monitor
        plugin.  See the documentation for 'scalyr_logging.AgentLogger.emit_value' method for more details.
        """
        try:
            self._logger.info('Starting monitor')
            while not self._is_stopped():
                try:
                    self.gather_sample()
                except Exception:
                    self._logger.exception('Failed to gather sample due to the following exception')
                    self.increment_counter(errors=1)

                self._sleep_but_awaken_if_stopped(self._sample_interval_secs)
            self._logger.info('Monitor has finished')
        except Exception:
            self._logger.exception('Monitor died from due to exception:', error_code='failedMonitor')

    def gather_sample(self):
        """Derived classes should implement this method to gather a data sample for this monitor plugin
        and report it.

        If the default "run" method implementation is not overridden, then
        derived classes must implement this method to actual perform the
        monitor-specific work to gather whatever information it should be
        collecting.

        It is expected that the derived class will report any gathered metric samples
        by using the 'emit_value' method on self._logger.  They may invoke that method
        multiple times in a single 'gather_sample' call to report multiple metrics.
        See the documentation for 'scalyr_logging.AgentLogger.emit_value' method for more details.

        Any exceptions raised by this method will be reported as an error but will
        not stop execution of the monitor.
        """
        pass

    def set_sample_interval(self, secs):
        """Sets the number of seconds between calls to gather_sample when running.

        This must be invoked before the monitor is started.

        Params:
            secs:  The number of seconds, which can be fractional.
        """
        self._sample_interval_secs = secs

    def open_metric_log(self):
        try:
            self._logger.openMetricLogForMonitor(self.log_config['path'], self)
            self.__metric_log_open = True
            return True
        except Exception:
            self._logger.exception('Failed to open metric log', error_code='failedMetricLog')
            return False

    def close_metric_log(self):
        if self.__metric_log_open:
            self._logger.closeMetricLog()
            self.__metric_log_open = False

    def _is_stopped(self):
        """Returns whether or not the "stop" method has been invoked."""
        return not self._run_state.is_running()

    def _sleep_but_awaken_if_stopped(self, time_to_sleep):
        """Sleeps for the specified amount of seconds or until the stop() method on this instance is invoked, whichever
         comes first.

        Arguments:
            time_to_sleep:  The number of seconds to sleep.

        Returns:
            True if the stop() has been invoked.
        """
        return self._run_state.sleep_but_awaken_if_stopped(time_to_sleep)


class MonitorConfig(object):
    """Encapsulates configuration parameters for a single monitor instance and includes helper utilities to
    validate configuration values.

    This supports most of the operators and methods that dict supports, but has additional support to allow
    Monitor developers to easily validate configuration values.  See the get method for more details.

    This abstraction does not support any mutator operations.  The configuration is read-only.
    """
    def __init__(self, content=None):
        """Initializes MonitorConfig.

        Arguments:
            content:  A dict containing the key/values pairs to use.
        """
        self.__map = {}
        if content is not None:
            for x in content:
                self.__map[x] = content[x]

    def __len__(self):
        """Returns the number of keys in the JsonObject"""
        return len(self.__map)

    def get(self, field, required_field=False, max_allowed_value=None, min_allowed_value=None,
            convert_to=None, default=None):
        """Returns the value for the requested field.

        This method will optionally apply some validation rules as indicated by the optional arguments.  If any
        of these validation operations fail, then a BadMonitorConfiguration exception is raised.  Monitor developers are
        encouraged to catch this exception at their layer.

        Arguments:
            field:  The name of the field.
            required_field:  If True, then will raise a BadMonitorConfiguration exception if the
                field is not present.
            convert_to:  If not None, then will convert the value for the field to the specified
                type.  Only int, bool, float, long, str, and unicode are supported.  If the type
                conversion cannot be done, a BadMonitorConfiguration exception is raised.
                The only true conversions allowed are those from str, unicode value to other types
                such as int, bool, long, float.  Trivial conversions are allowed from int, long to float,
                but not the other way around.  Additionally, any primitive type can be converted to str, unicode.
            default:  The value to return if the field is not present in the configuration.
            max_allowed_value:  If not None, the maximum allowed value for field.  Raises a BadMonitorConfiguration
                if the value is greater.
            min_allowed_value:  If not None, the minimum allowed value for field.  Raises a BadMonitorConfiguration
                if the value is less than.

        Returns the value.
        """
        if required_field and field not in self.__map:
            raise BadMonitorConfiguration('Missing required field "%s"' % field, field)
        result = self.__map.get(field, default)

        if result is None:
            return result

        if convert_to is not None and type(result) != convert_to:
            result = self.__perform_conversion(field, result, convert_to)

        if max_allowed_value is not None and result > max_allowed_value:
            raise BadMonitorConfiguration('Field "%s" value of %s is greater than the max allowed %s' % (
                                          field, str(result), str(max_allowed_value)), field)

        if min_allowed_value is not None and result < min_allowed_value:
            raise BadMonitorConfiguration('Field "%s" value of %s is less than the min allowed %s' % (
                                          field, str(result), str(min_allowed_value)), field)

        return result

    def __perform_conversion(self, field_name, value, convert_to):
        value_type = type(value)
        primitive_types = (int, long, float, str, unicode, bool)
        if convert_to not in primitive_types:
            raise Exception('Unsupported type for conversion passed as convert_to: "%s"' % str(convert_to))
        if value_type not in primitive_types:
            raise BadMonitorConfiguration('Unable to convert type %s for field "%s" to type %s' % (
                str(value_type), field_name, str(convert_to)), field_name)

        # Anything is allowed to go to str/unicode
        if convert_to == str or convert_to == unicode:
            return convert_to(value)

        # Anything is allowed to go from string/unicode to the conversion type, as long as it can be parsed.
        # Handle bool first.
        if value_type in (str, unicode):
            if convert_to == bool:
                return str(value).lower() == 'true'
            elif convert_to in (int, float, long):
                try:
                    return convert_to(value)
                except ValueError:
                    raise BadMonitorConfiguration('Could not parse value %s for field "%s" as numeric type %s' % (
                                                  value, field_name, str(convert_to)), field_name)

        if convert_to == bool:
            raise BadMonitorConfiguration('A numeric value %s was given for boolean field "%s"' % (
                                          str(value), field_name), field_name)

        # At this point, we are trying to convert a number to another number type.  We only allow long to int,
        # and long, int to float.
        if convert_to == float and value_type in (long, int):
            return float(value)
        if convert_to == long and value_type == int:
            return long(value)

        raise BadMonitorConfiguration('A numeric value of %s was given for field "%s" but a %s is required.', (
                                      str(value), field_name, str(convert_to)))

    def __iter__(self):
        return self.__map.iterkeys()

    def iteritems(self):
        """Returns an iterator over the items (key/value tuple) for this object."""
        return self.__map.iteritems()

    def itervalues(self):
        """Returns an iterator over the values for this object."""
        return self.__map.itervalues()

    def iterkeys(self):
        """Returns an iterator over the keys for this object."""
        return self.__map.iterkeys()

    def items(self):
        """Returns a list of items (key/value tuple) for this object."""
        return self.__map.items()

    def values(self):
        """Returns a list of values for this object."""
        return self.__map.values()

    def keys(self):
        """Returns a list keys for this object."""
        return self.__map.keys()

    def __getitem__(self, field):
        if not field in self:
            raise KeyError('The missing field "%s" in monitor config.' % field)
        return self.__map[field]

    def copy(self):
        result = MonitorConfig()
        result.__map = self.__map.copy()
        return result

    def __contains__(self, key):
        """Returns True if the JsonObject contains a value for key."""
        return key in self.__map

    def __eq__(self, other):
        if other is None:
            return False
        if type(self) is not type(other):
            return False
        assert isinstance(other.__map, dict)
        return self.__map == other.__map

    def __ne__(self, other):
        return not self.__eq__(other)


class BadMonitorConfiguration(Exception):
    """Exception indicating a bad monitor configuration, such as missing a required field."""
    def __init__(self, message, field):
        self.field = field
        Exception.__init__(self, message)