#!/usr/bin/env python
# Copyright 2014, Scalyr, Inc.
#
# Contains the data structures used to represent a snapshot of the
# agent's status, giving such details as the number of log bytes copied,
# whether or not the configuration file was successfully parsed, etc.
#
# These data structures are generated by the ScalyrAgent's __generate_status
# method and are used to create both the interactive status (which is emitted when
# the 'status -v' command is invoked by the user) and the status periodically recorded
# in the agent log.


class AgentStatus(object):
    """The main status container object, holding references to all other status elements.
    """
    def __init__(self):
        # The time (in seconds past epoch) when the agent process was launched.
        self.launch_time = None
        # The user name the agent process is running under.
        self.user = None
        # The version string for the agent.
        self.version = None
        # The name of the host the agent is running on.
        self.server_host = None
        # The URL of the scalyr server that the agent is connected to (such as https://www.scalyr.com/).
        self.scalyr_server = None
        # The path for the agent's log file.
        self.log_path = None
        # The ConfigStatus object recording the status for the configuration file.
        self.config_status = None
        # The CopyingManagerStatus object recording the status of the log copying manager (or none if CopyingManager
        # has not been started). This contains information about the different log paths being watched and the
        # progress of copying their bytes.
        self.copying_manager_status = None
        # The MonitorManagerStatus object recording the status of the monitor manager (or none if the MonitorManager
        # has not been started).  This contains information about the different ScalyrMonitors being run.
        self.monitor_manager_status = None

class OverallStats(object):
    """Used to track stats that are calculated over the lifetime of the agent.
    """
    def __init__(self):
        # The time in seconds past epoch when the agent was started.
        self.launch_time = None
        # The version string for the agent.
        self.version = None
        # The current number of paths the log copier is watching.
        self.num_watched_paths = 0
        # The current number of file paths the log copier is copying.
        self.num_copying_paths = 0
        # The current number of running monitors.
        self.num_running_monitors = 0
        # The current number of monitors that should be running but are not.
        self.num_dead_monitor = 0
        # The total amount of user time CPU used by the agent (cpu secs).
        self.user_cpu = 0
        # The total amount of system time CPU used by the agent (cpu secs)
        self.system_cpu = 0
        # The current resident size in bytes of the agent process.
        self.rss_size = 0

        # The total number of log bytes copied to the Scalyr servers.
        self.total_bytes_copied = 0
        # The total number of log bytes that were skipped and were not considered to be sent to the Scalyr servers.
        self.total_bytes_skipped = 0
        # The total number of log bytes that were not sent to the Scalyr servers due to subsampling rules.
        self.total_bytes_subsampled = 0
        # The total number of log bytes that were not sent to Scalyr due to errors on either the client or server side.
        self.total_bytes_failed = 0
        # The total number of redactions that were applied to the log lines before being sent to the Scalyr servers.
        self.total_redactions = 0
        # The total number of errors seen when issuing a copy request.
        self.total_copy_requests_errors = 0
        # The total number of lines reported by monitors.
        self.total_monitor_reported_lines = 0
        # The total number of errors seen by executing monitors.
        self.total_monitor_errors = 0

    def __add__(self, other):
        """Adds all of the 'total_' fields of this instance and other together and returns a new OverallStats containing
        the result.
        """
        result = OverallStats()
        result.total_bytes_copied = self.total_bytes_copied + other.total_bytes_copied
        result.total_bytes_skipped = self.total_bytes_skipped + other.total_bytes_skipped
        result.total_bytes_subsampled = self.total_bytes_subsampled + other.total_bytes_subsampled
        result.total_bytes_failed = self.total_bytes_failed + other.total_bytes_failed
        result.total_redactions = self.total_redactions + other.total_redactions
        result.total_copy_requests_errors = self.total_copy_requests_errors + other.total_copy_requests_errors
        result.total_monitor_reported_lines = self.total_monitor_reported_lines + other.total_monitor_reported_lines
        result.total_monitor_errors = self.total_monitor_errors + other.total_monitor_errors

        return result


class ConfigStatus(object):
    """The status pertaining to parsing of the configuration file."""
    def __init__(self):
        # The path of the configuration file.
        self.path = None
        # The paths for additional configuration files read from the config directory.
        self.additional_paths = []
        # The last time the configuration file changed and was read by this agent.
        self.last_read_time = None
        # A status line describing if the configuration file was successfully parsed.
        self.status = None
        # If the status file count not be parsed/used, a string describing the error.
        self.last_error = None
        # The last time the configuration file was successfully parsed.
        self.last_good_read = None
        # The last time the agent checked to see if the configuration file has changed.
        self.last_check_time = None


class CopyingManagerStatus(object):
    """The status object containing information about the agent's copying components."""
    def __init__(self):
        # The total number of bytes successfully uploaded.
        self.total_bytes_uploaded = 0
        # The last time the agent successfully copied bytes from log files to the Scalyr servers.
        self.last_success_time = None
        # The last time the agent attempted to copy bytes from log files to the Scalyr servers.
        self.last_attempt_time = None
        # The size of the request for the last attempt.
        self.last_attempt_size = None
        # The last response from the Scalyr servers.
        self.last_response = None
        # The last status from the last response (should be 'success').
        self.last_response_status = None
        # The total number of failed copy requests.
        self.total_errors = None

        # LogMatcherStatus objects for each of the log paths being watched for copying.
        self.log_matchers = []


class LogMatcherStatus(object):
    """The status object containing information about all of the copying being performed for a particular
    log path including globbing."""
    def __init__(self):
        # The path.
        self.log_path = None
        # True if the log path contains globbing characters.
        self.is_glob = None
        # The last time the agent checked the path for new matches.
        self.last_check_time = None
        # For any matching file paths, a LogProcessorStatus object describing the copying.
        self.log_processors_status = []


class LogProcessorStatus(object):
    """The status object containing information about the progress of the bytes being copied for a particular
    file."""
    def __init__(self):
        # The path of the file (will not contain glob characters).  This will be a path to an existing file.
        self.log_path = None
        # The last time the file was checked for new bytes.
        self.last_scan_time = None
        # The total bytes copied to the Scalyr servers.
        self.total_bytes_copied = 0
        # The number of bytes that still need to be sent to the Scalyr servers.
        self.total_bytes_pending = 0
        # The total bytes that were skipped (due to the log lines being too old, or the agent falling behind).
        self.total_bytes_skipped = 0
        # The total bytes that failed due to errors at either the server or client.
        self.total_bytes_failed = 0
        # The total bytes that were not sent to the server due to subsampling rules.
        self.total_bytes_dropped_by_sampling = 0
        # The total number of log lines copied to the Scalyr servers.
        self.total_lines_copied = 0
        # The total number of log lines that were not sent to the server due to subsampling rules.
        self.total_lines_dropped_by_sampling = 0
        # The total number of redactions applied to the log lines copied to the server.
        self.total_redactions = 0


class MonitorManagerStatus(object):
    """The status object containing information about all of the running monitors."""
    def __init__(self):
        # The total number of monitors that are currently running.
        self.total_alive_monitors = 0
        # The MonitorStatus object for each monitor that is currently running or should be running.
        self.monitors_status = []


class MonitorStatus(object):
    """The status object for a specific instance of a ScalyrMonitor."""
    def __init__(self):
        # The name of the monitor.
        self.monitor_name = None
        # The total number of metric lines reported by the monitor.
        self.reported_lines = 0
        # The total number of errors produced by the monitor.
        self.errors = 0
        # Whether or not the monitor is running.
        self.is_alive = False