# coding: utf-8
#
# Copyright (C) 2012  Niklas Rosenstein
# <rosensteinniklas@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import scan.base
import string
import cStringIO

class Cursor(scan.base.Copiable):
    """ This class serves as a container for storing the position, column and
        line numbers for the scanner. It additionally manages the counting
        of these.

        .. attribute:: position

        .. attribute:: column

        .. attribute:: line
        """

    def __init__(self, position=0, column=0, line=0):
        self.position = position
        self.column = column
        self.line = line

    def __repr__(self):
        return 'Cursor(%d, %d, %d)' % (self.position, self.column, self.line)

    def process(self, character):
        """ This method is called from the scanner and is intended to increase
            the position, column and line counts respectively. """

        if character == '\n':
            self.line += 1
            self.column = 0
        else:
            self.column += 1

        self.position += 1

  # scan.base.Copiable

    def copy_from_instance(self, other):
        self.position = other.position
        self.column = other.column
        self.line = other.line
        return super(Cursor, self).copy_from_instance(other)

class Scanner(object):
    """ This class implements simple scanning mechanisms for file-like objects.

        .. attribute:: source
            The file-like object that is used to read from.

        .. attribute:: char
            The current character from the source. `None` after initialization
            and `''` at EOF.
        """

    @classmethod
    def from_string(self, string):
        """ Create and initialize a :class:`Scanner` instance from a string
            by wrapping a :class:`cStringIO.StringIO` object around. This
            function raises `TypeError` when *string* is not a `basestring`
            instance. """
        if not isinstance(string, basestring):
            raise TypeError('expected basestring instance')
        file = cStringIO.StringIO(string)
        return self(file)

    @classmethod
    def from_filename(self, filename, binary=False):
        """ Create and initialize a :class:`Scanner` instance from a string
            by opening a file-object via the built-in :func:`open`. If
            *binary* is *True*, the file is opened with the `wb` flag, only
            `w` is used. """
        if not isinstance(string, basestring):
            raise TypeError('expected basestring instance.')
        if binary:
            openmode = 'wb'
        else:
            openmode = 'w'
        file = open(filename, openmode)
        return self(file)


    def __init__(self, source, cursor=None):
        """ Initialize the instance with a file-like object. """
        self.source = source
        self._cursor = cursor or Cursor()
        self.char = None

    def __iter__(self):
        while self.char:
            yield self.char
            self.read()

    @property
    def cursor(self):
        return self._cursor.copy()

    @cursor.setter
    def cursor(self, cursor):
        self.seek_to(cursor)

    def seek_to(self, cursor):
        """ Seek to the position defined in the cursor. The column and line
            number are taken as they are, but they are not needed for seeking.
            They should be kept synchronized to prevent from creating invalid
            or wrong information.

            When `cursor.position` is zero, the `char` attribute will be
            `None` again, as it was after creating the Scanner instance. """

        if not isinstance(cursor, Cursor):
            raise TypeError('cursor instance expected')

        offset = self.source.tell() - self._cursor.position
        position = offset + cursor.position

        if position <= 0 or cursor.position <= 0:
            self.source.seek(position)
            self.char = None
        else:
            self.source.seek(position - 1)
            self.char = self.source.read(1)

        self._cursor = cursor.copy()

    def read(self):
        """ Read the next character from the source file. Increases line-
            and column-count respectively. """
        self.char = self.source.read(1)
        self._cursor.process(self.char)
        return self.char

    def read_set(self, set, default='', fallback=False, maxlen=-1):
        """ Read out the longest sequence of characters in a row where all
            of the characters appear in the iterable *set*. If *maxlen* is
            greater or equal zero, not more characters than the number passed
            here are read.

            If *fallback* is set to `True`, the scanner will return to its 
            original position. Returns *default* when no characters were read.
            """

        cursor = self.cursor
        string = ''
        while self.char and self.char in set:
            if maxlen >= 0 and len(string) > maxlen:
                break
            else:
                string += self.char
                self.read()

        if fallback:
            self.seek_to(cursor)

        return string or default

    def skip_set(self, set):
        """ Skip all characters from the current position that appear in the
            iterable *set*. The number of skipped characters is returned. """
        count = 0
        while self.char and self.char in set:
            count += 1
            self.read()
        return count

    def match(self, sequence, default=None, fallback=False):
        """ Matches all characters from the current position in the same order
            with the elements appearing in the iterable *sequence*.

            *fallback* will make the scanner returns to its original position
            in the source from where matching started. The Scanner automatically
            returns to the original position when no the sequence did **not
            match**.
            If the sequence did not match, *default* is returned, else the
            matched sequence.
            """

        cursor = self.cursor
        string = ''
        matched = True
        for subset in sequence:
            if not self.char or not self.char in subset:
                matched = False
                break
            else:
                string += self.char
                self.read()

        if not matched or fallback:
            self.seek_to(cursor)

        return string if matched else default


