# coding: utf-8
#
# Copyright (c) 2012-2013, Niklas Rosenstein
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met: 
# 
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer. 
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in
#    the documentation and/or other materials provided with the
#    distribution. 
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
# FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
# COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
# BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
# ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
# 
# The views and conclusions contained in the software and
# documentation are those of the authors and should not be interpreted
# as representing official policies,  either expressed or implied, of
# the FreeBSD Project.

import scan.base
import string
import cStringIO

class Cursor(scan.base.Copiable):
    """
    This class serves as a container for storing the position, column and
    line numbers for the scanner. It additionally manages the counting
    of these.

    .. attribute:: position

    .. attribute:: column

    .. attribute:: line

    .. attribute:: cr_newline
        When this is True, the Cursor will treat the carriage-return
        character as a new-line character. When False, it will be treated
        like *any* other character. Default is True.
    """

    def __init__(self, position=0, column=0, line=1, cr_newline=True):
        self.position = position
        self.column = column
        self.line = line
        self.cr_newline = cr_newline
        self.prev_char = None

    def __repr__(self):
        return 'Cursor(pos: %d, col: %d, line: %d)' % (
               self.position, self.column, self.line)

    def process(self, character):
        """
        This method is called from the scanner and is intended to increase
        the position, column and line counts respectively.
        """

        if not character and not self.prev_char:
            return
        self.prev_char = character

        if self.is_newline(character):
            self.line += 1
            self.column = 0
        else:
            self.column += 1

        self.position += 1

    def is_newline(self, character):
        return character == '\n' or (self.cr_newline and character == '\r')

  # scan.base.Copiable

    def copy_from_instance(self, other):
        self.position = other.position
        self.column = other.column
        self.line = other.line
        self.cr_newline = other.cr_newline
        self.prev_char = other.prev_char
        return super(Cursor, self).copy_from_instance(other)

class Scanner(object):
    """
    This class implements simple scanning mechanisms for file-like objects.

    .. attribute:: source
        The file-like object that is used to read from.

    .. attribute:: char
        The current character from the source. `None` after initialization
        and `''` at EOF.
    """

    @classmethod
    def from_string(self, string):
        """
        Create and initialize a :class:`Scanner` instance from a string
        by wrapping a :class:`cStringIO.StringIO` object around. This
        function raises `TypeError` when *string* is not a `basestring`
        instance.
        """

        if not isinstance(string, basestring):
            raise TypeError('expected basestring instance')
        file = cStringIO.StringIO(string)
        return self(file)

    @classmethod
    def from_filename(self, filename, binary=True):
        """
        Create and initialize a :class:`Scanner` instance from a string
        by opening a file-object via the built-in :func:`open`. If
        *binary* is *True*, the file is opened with the `rb` flag, only
        `r` is used.

        .. note:: When using non-binary mode the file-handler will convert
                  any newline-sequence to ``'\n'`` but the file-cursor
                  will jump about two if CRLF is used which *will* cause
                  trouble when seeking !!
        """

        if not isinstance(filename, basestring):
            raise TypeError('expected basestring instance.')
        if binary:
            openmode = 'rb'
        else:
            openmode = 'r'
        file = open(filename, openmode)
        return self(file)

    def __init__(self, source, cursor=None):
        """ Initialize the instance with a file-like object. """
        self.source = source
        self._cursor = cursor or Cursor()
        self.char = None

    def __iter__(self):
        while self.char:
            yield self.char
            self.read()

    def seek_to(self, cursor):
        """
        Seek to the position defined in the cursor. The column and line
        number are taken as they are, but they are not needed for seeking.
        They should be kept synchronized to prevent from creating invalid
        or wrong information.

        When `cursor.position` is zero, the `char` attribute will be
        `None` again, as it was after creating the Scanner instance.
        """

        if not isinstance(cursor, Cursor):
            raise TypeError('cursor instance expected')

        offset = self._cursor.position - self.source.tell()
        position = offset + cursor.position
        if position == self.source.tell():
            return

        char = self.char

        if position <= 0 or cursor.position <= 0:
            self.source.seek(0)
            self.char = None
        else:
            self.source.seek(position - 1)
            self.char = self.source.read(1)

        self._cursor = cursor.copy()

    def read(self):
        """
        Read the next character from the source file. Increases line-
        and column-count respectively.
        """

        self.char = self.source.read(1)
        self._cursor.process(self.char)
        return self.char

    def read_set(self, set, default='', fallback=False, maxlen=-1):
        """
        Read out the longest sequence of characters in a row where all
        of the characters appear in the iterable *set*. If *maxlen* is
        greater or equal zero, not more characters than the number passed
        here are read.

        If *fallback* is set to `True`, the scanner will return to its 
        original position. Returns *default* when no characters were read.
        """

        cursor = self.cursor
        string = ''
        while self.char and self.char in set:
            if maxlen >= 0 and len(string) > maxlen:
                break
            else:
                string += self.char
                self.read()

        if fallback:
            self.seek_to(cursor)

        return string or default

    def read_line(self):
        value = ''
        while self.char and not self.cursor.is_newline(self.char):
            value += self.char
            self.read()
        return value

    def skip_set(self, set):
        """
        Skip all characters from the current position that appear in the
        iterable *set*. The number of skipped characters is returned. """
        count = 0
        while self.char and self.char in set:
            count += 1
            self.read()
        return count

    def match(self, sequence, default=None, fallback=False,
                    charprocessor=lambda x: x):
        """
        Matches all characters from the current position in the same order
        with the elements appearing in the iterable *sequence*.

        *fallback* will make the scanner returns to its original position
        in the source from where matching started. The Scanner automatically
        returns to the original position when no the sequence did **not
        match**.
        If the sequence did not match, *default* is returned, else the
        matched sequence.
        """

        cursor = self.cursor
        string = ''
        matched = True
        for subset in sequence:
            if not self.char or not charprocessor(self.char) in subset:
                matched = False
                break
            else:
                string += self.char
                self.read()

        if not matched or fallback:
            self.seek_to(cursor)

        return string if matched else default

    @property
    def cursor(self):
        return self._cursor.copy()

    @cursor.setter
    def cursor(self, cursor):
        self.seek_to(cursor)

