"""A scheduler for periodic jobs that uses the builder pattern to
construct easily readable definitions for periodic tasts.

Examples:
    every(10).minutes.do(job) -- Run job() every 10 minutes
    every().hour.do(job) -- Run job() every hour
    every().day.at('10:30').do(job) -- Run job() every day at 10:30

Additional reading:
    http://adam.heroku.com/past/2010/6/30/replace_cron_with_clockwork/
    https://devcenter.heroku.com/articles/scheduled-jobs-custom-clock-processes
    https://devcenter.heroku.com/articles/clock-processes-python
"""
import datetime
import logging
import time

# All registered jobs.
_jobs = []


def every(interval=1):
    job = PeriodicJob(interval)
    _jobs.append(job)
    return job


def tick():
    """Run all jobs that are scheduled to run.

    Please note that it is *intended behavior that tick() does not run
    missed jobs*. For example, if you've registered a job that should
    run every minute and you only call tick() in one hour increments
    then your job won't be run 60 times in between but only once.
    """
    logging.debug('Ticking clock')
    for job in _jobs:
        if job.should_run:
            job.run()


def print_jobs():
    for job in sorted(_jobs):
        logging.info('%s', job)


def run_all_jobs(delay=60):
    """Run all jobs regardless if they are scheduled to run or not.

    A delay of `delay` seconds is added between each job. This can help
    to distribute the system load generated by the jobs more evenly over
    time."""
    logging.info('Running *all* %i jobs with %is delay inbetween',
                 len(_jobs), delay)
    for job in _jobs:
        job.run()
        time.sleep(delay)

def clear_all_jobs():
    global _jobs
    _jobs = []


class PeriodicJob(object):
    def __init__(self, interval):
        self.interval = interval
        self.func = None
        self.unit = None
        self.time = None
        self.last_run = None
        self.next_run = None
        self.period = None

    def __str__(self):
        def format_time(t):
            return t.strftime("%Y-%m-%d %H:%M:%S") if t else '[never]'

        timestats = '(last run: %s, next run: %s)' % (
                    format_time(self.last_run), format_time(self.next_run))

        if self.time is not None:
            return 'Every %s %s at %s do %s %s' % (
                self.interval,
                self.unit[:-1] if self.interval == 1 else self.unit,
                self.time, self.func.__name__, timestats)
        else:
            return 'Every %s %s do %s %s' % (
                self.interval,
                self.unit[:-1] if self.interval == 1 else self.unit,
                self.func.__name__, timestats)

    @property
    def second(self):
        return self.seconds

    @property
    def seconds(self):
        self.unit = 'seconds'
        return self

    @property
    def minute(self):
        return self.minutes

    @property
    def minutes(self):
        self.unit = 'minutes'
        return self

    @property
    def hour(self):
        return self.hours

    @property
    def hours(self):
        self.unit = 'hours'
        return self

    @property
    def day(self):
        return self.days

    @property
    def days(self):
        self.unit = 'days'
        return self

    @property
    def week(self):
        return self.weeks

    @property
    def weeks(self):
        self.unit = 'weeks'
        return self

    def at(self, time_str):
        assert self.unit == 'days'
        hour, minute = [int(t) for t in time_str.split(':')]
        assert 0 <= hour <= 23
        assert 0 <= minute <= 59
        self.time = datetime.time(hour, minute)
        return self

    def do(self, func):
        self.func = func
        self.compute_next_run()
        return self

    @property
    def should_run(self):
        return datetime.datetime.now() >= self.next_run

    def run(self):
        logging.info('Running job %s', self)
        self.func()
        self.last_run = datetime.datetime.now()
        self.compute_next_run()

    def compute_next_run(self):
        if self.unit == 'seconds':
            self.period = datetime.timedelta(seconds=self.interval)
        elif self.unit == 'minutes':
            self.period = datetime.timedelta(minutes=self.interval)
        elif self.unit == 'hours':
            self.period = datetime.timedelta(hours=self.interval)
        elif self.unit == 'days':
            if self.time is None:
                self.period = datetime.timedelta(days=self.interval)
            else:
                next_run = (datetime.datetime.today() +
                            datetime.timedelta(days=self.interval))
                self.next_run = next_run.replace(hour=self.time.hour,
                                                 minute=self.time.minute,
                                                 second=self.time.second,
                                                 microsecond=0)
        elif self.unit == 'weeks':
            self.period = datetime.timedelta(weeks=self.interval)
        else:
            raise ValueError('Invalid unit: %s' % self.unit)
        if self.period:
            self.next_run = datetime.datetime.now() + self.period
