# -*- coding: utf-8 -*-
'''
Created on 13/giu/2011

@author: simone cittadini
'''
from string import capwords

from django.db import models
from django.db.models.signals import m2m_changed
from django.dispatch import receiver

from alara.core.exceptions import IncoerenzaNelleMaschere


MAX_PREFIX_LENGTH = 16

DEST_TYPES_CHOICES = ((1,'FIX'), (2,'MOB'), (3,'NGN'))
DEST_TYPES = {'FIX' : 1, 'MOB' : 2, 'NGN': 3}

@receiver(m2m_changed)
def callback_m2m_changed(sender, **kwargs):
    if str(sender) == "<class 'alara.prefissi.models.NomeDistretto_prefissi'>":
        if kwargs['action'] == 'pre_add':
            # La maschera associata non può contenere prefissi duplicati;
            #  prefissi già esistenti sotto la stessa maschera vengono rimossi
            #  dall'attuale nome distretto.
            maschera = kwargs['instance'].maschera
            for pk in kwargs['pk_set']:
                prefisso = Prefisso.objects.get(pk=pk)
                for nome_distretto in prefisso.nomi_distretto.all():
                    if nome_distretto.maschera == maschera:
                        nome_distretto.prefissi.remove(prefisso)


class Maschera(models.Model):
    """
    Raccolta di associazioni fra prefissi e distretti.
    """
    nome = models.CharField(max_length=32, unique=True)
    base = models.BooleanField(default=False)

    class Meta:
        ordering = ('base', 'nome',)

    def __unicode__(self):
        return self.nome

    def save(self, *args, **kwargs):
        # Non può esistere più di una maschera base
        if self.base:
            if Maschera.objects.filter(
                                base__exact=True).exclude(
                                nome__exact=self.nome).count() == 1:
                raise IncoerenzaNelleMaschere(
                                    "Può esistere solo una maschera base")
        super(Maschera, self).save(*args, **kwargs)

    @staticmethod
    def get_maschera_base():
        try:
            return Maschera.objects.get(base=True)
        except Maschera.DoesNotExist:
            return None


class Prefisso(models.Model):
    """
    Rappresentazione di un prefisso internazionale.
    """
    pre = models.CharField(max_length=16, unique=True)

    class Meta:
        ordering = ('pre',)

    def __unicode__(self):
        return self.pre

    def save(self, *args, **kwargs):
        if self.pre == None or len(self.pre) < 2:
            raise IncoerenzaNelleMaschere("Prefisso troppo corto")
        super(Prefisso, self).save(*args, **kwargs)

    @staticmethod
    def get_prefisso(dst, min_length=2, max_length=MAX_PREFIX_LENGTH):
        if len(dst) < max_length:
            max_length = len(dst)
        for i in range(min_length, max_length):
            try:
                p = Prefisso.objects.get(pre=dst[0:max_length - i])
                return p
            except:
                pass
        raise  # TODO 

    def get_distretto_base(self):
        maschera_base = Maschera.get_maschera_base()
        try:
            return self.nomi_distretto.get(maschera=maschera_base)
        except:
            return None


class NomeDistretto(models.Model):
    """
    Associazione fra un distretto / carrier e un insieme di prefissi.
    """
    nome = models.CharField(max_length=64)
    tipo = models.CharField(max_length=3, choices=DEST_TYPES_CHOICES)
    prefissi = models.ManyToManyField(Prefisso, related_name='nomi_distretto')
    maschera = models.ForeignKey(Maschera)

    class Meta:
        ordering = ('maschera', 'nome',)
        unique_together = ('nome', 'maschera',)

    def __unicode__(self):
        return "%s - %s" % (self.maschera, self.nome)

    def save(self, *args, **kwargs):
        self.nome = capwords(self.nome.strip())
        super(NomeDistretto, self).save(*args, **kwargs)
