# -*- coding: utf-8 -*-
'''
Created on 13/giu/2011

@author: simone cittadini
'''
from django.db import models
from django.conf import settings

from sctel.prefissi.models import NomeDistretto


PROTOCOLLI = (
    ("SIP", "SIP"),
    ("ZAP", "ZAP"),
    ("IAX", "IAX"),
    ("323", "323"),
)


class Linea(models.Model):
    """
    Classe base per i tipi di linea.
    """
    nome = models.CharField(max_length=32, unique=True)

    def __unicode__(self):
        return "%s" % self.nome

    @staticmethod
    def get_ingresso(dst, len_tech_prefix=None):
        """
        Dato un numero chiamato ne estrae il tech prefix e ritorna la
        linea corrispondente.
        """
        if len_tech_prefix:
            try:
                t = TechPrefix.objects.get(codice=dst[0:len_tech_prefix])
                return t.ingresso
            except TechPrefix.DoesNotExist:
                raise  ## TODO
        else:
            for i in range(len(dst)):
                try:
                    t = TechPrefix.objects.get(codice=dst[0:i])
                except:
                    pass
        raise  ## TODO

    def get_terminazione(self, distretto):
        """
        Ritorna l'ultima linea della rotta.
        """
        raise NotImplementedError
    
    def get_backup(self, distretto):
        """
        Per una linea di rivendita ritorna una terminazione alternativa,
        altrimenti ritorna None.
        """
        raise NotImplementedError
    
    def get_dial_string(self, dst_number, pre=None):
        """
        Per una linea di Terminazione ritorna gli argomenti per un comando dial
        dello switch, relativi a uno dei peer disponibili.
        """
        raise NotImplementedError


class Terminazione(Linea):
    """
    Rappresenta una terminazione fisica su un peer.
    """
    class Meta:
        verbose_name_plural = 'Terminazioni'
        
    def get_terminazione(self, distretto):
        return self

    def get_dial_string(self, dst_number, pre=None):
        try:
            for p in Peer.objects.filter(terminazione__exact=self).order_by('?'):
                if p.protocollo == 'SIP':
                    if p.secret and p.username:
                        if p.strip_local_countrycode and dst_number[0:4] == getattr(settings, 'LOCAL_COUNTRYCODE', '0039'):
                            dst_number = dst_number[4:]
                        if p.strip_leading_zeroes and dst_number[0:2] == '00':
                            dst_number = dst_number[2:]
                        return 'SIP/%s:%s@%s:%s/%s%s' % (p.username, p.secret, p.address, p.port, p.tech_prefix, dst_number)
        except:
            raise 


class Rivendita(Linea):
    """
    Rappresenta una scelta verso terminazione data dall'associazione
    fra ingresso e distretto.
    """
    def get_terminazione(self, distretto):
        try:
            r = Rotta.objects.get(
                                linea_in=self,
                                distretto=distretto)
            return r.get_terminazione()
        except Rotta.DoesNotExist:
            raise

    def get_backup(self, distretto):
        try:
            r = Rotta.objects.get(
                                linea_in=self,
                                distretto=distretto)
            return r.get_backup()
        except Rotta.DoesNotExist:
            raise


class Ingresso(Linea):
    """
    Rappresenta il punto di partenza di una chiamata.
    """
    def get_terminazione(self, distretto):
        try:
            r = Rotta.objects.get(linea_in=self,
                                distretto=distretto)
            return r.get_terminazione()
        except Rotta.DoesNotExist:
            raise


class Rotta(models.Model):
    linea_in = models.ForeignKey(Linea, related_name='rotte_dell_ingresso')
    linea_out = models.ForeignKey(Linea, related_name='rotte_dell_uscita')
    linea_backup = models.ForeignKey(Linea, related_name='rotte_del_backup',
                                     blank=True, null=True)
    nome_distretto = models.ForeignKey(NomeDistretto,
                                       related_name='rotte_del_distretto')

    def save(self, *args, **kwargs):
        # Il distretto deve essere della machera base
        # Il backup può esistere solo su una linea di rivendita
        # le combinazioni possono essere solo :
        # Ingresso -> Rivendita
        # Rivendita -> Terminazione
        pass

    def get_terminazione(self):
        return self.linea_out.get_terminazione()

    def get_backup(self):
        return self.linea_backup.get_backup()


class TechPrefix(models.Model):
    codice = models.CharField(max_length=12, unique=True)
    ingresso = models.ForeignKey(Ingresso)

    def __unicode__(self):
        return "%s-%s" % (self.ingresso, self.codice)

    class Meta:
        ordering = ['ingresso', 'codice']
        

class Peer(models.Model):
    protocollo = models.CharField(max_length=3, choices=PROTOCOLLI,
                                  default='SIP')
    address = models.CharField(max_length=64)
    port = models.CharField(max_length=6, default='5060')
    username = models.CharField(max_length=64, blank=True, null=True)
    secret = models.CharField(max_length=64, blank=True, null=True)
    strip_leading_zeroes = models.BooleanField(default=True)
    strip_local_countrycode = models.BooleanField(default=False)
    tech_prefix = models.CharField(max_length=12, blank=True)
    max_calls = models.PositiveIntegerField(default=0)
    terminazione = models.ForeignKey(Terminazione)

    def __unicode__(self):
        return "%s-%s-%s" % (self.terminazione, self.protocollo, self.address)

    class Meta:
        ordering = ['terminazione', 'protocollo', 'address']
