"""Tests for plotting utilities."""
import numpy as np
from numpy.testing import assert_array_equal
import nose
from nose.tools import assert_equal, raises

from .. import utils


a_norm = np.random.randn(100)


def test_pmf_hist_basics():
    """Test the function to return barplot args for pmf hist."""
    out = utils.pmf_hist(a_norm)
    assert_equal(len(out), 3)
    x, h, w = out
    assert_equal(len(x), len(h))

    # Test simple case
    a = np.arange(10)
    x, h, w = utils.pmf_hist(a, 10)
    nose.tools.assert_true(np.all(h == h[0]))


def test_pmf_hist_widths():
    """Test histogram width is correct."""
    x, h, w = utils.pmf_hist(a_norm)
    assert_equal(x[1] - x[0], w)


def test_pmf_hist_normalization():
    """Test that output data behaves like a PMF."""
    x, h, w = utils.pmf_hist(a_norm)
    nose.tools.assert_almost_equal(sum(h), 1)
    nose.tools.assert_less_equal(h.max(), 1)


def test_pmf_hist_bins():
    """Test bin specification."""
    x, h, w = utils.pmf_hist(a_norm, 20)
    assert_equal(len(x), 20)


def test_ci_to_errsize():
    """Test behavior of ci_to_errsize."""
    cis = [[.5, .5],
           [1.25, 1.5]]

    heights = [1, 1.5]

    actual_errsize = np.array([[.5, 1],
                               [.25, 0]])

    test_errsize = utils.ci_to_errsize(cis, heights)
    assert_array_equal(actual_errsize, test_errsize)


def test_desaturate():
    """Test that we get the right return values from desat."""
    out1 = utils.desaturate("red", .5)
    assert_equal(out1, (1., .5, .5))

    out2 = utils.desaturate("#00FF00", .5)
    assert_equal(out2, (.5, 1., .5))

    out3 = utils.desaturate((0, 0, 1), .5)
    assert_equal(out3, (.5, .5, 1.))

    out4 = utils.desaturate("red", .5, "hls")
    assert_equal(out4, (.75, .25, .25))


@raises(ValueError)
def test_desaturation_pct():
    """Test that pct outside of [0, 1] raises exception."""
    utils.desaturate("blue", 50)


def test_saturate():
    """Test performance of saturation function."""
    out = utils.saturate((1, .5, .5))
    assert_equal(out, (1, 0, 0))


@raises(ValueError)
def test_hue_space():
    """Test that desaturation space choise is constrained."""
    utils._hue_space_params("cielab")


def test_saturation_index():
    """Test index to saturation channel for different spaces."""
    hsv, _, _ = utils._hue_space_params("hsv")
    assert_equal(hsv, 1)
    hls, _, _ = utils._hue_space_params("hls")
    assert_equal(hls, 2)
