"""

ABC
---

Abstract class for all components.

.. autoclass:: seapy.components.component.Component

"""

from ..base import Base, MaterialLink, LinkedList

import abc
import math
import cmath
import numpy as np
import logging
from weakref import WeakSet


class Component(Base):
    """ Abstract Base Class for components."""
    
    SORT = 'Component'

    material = MaterialLink()
    """
    Material which this component consists of.
    """
    
    _DEPENDENCIES = ['material']   # The right way to do it...
    
    linked_junctions = LinkedList()
    """
    junctions this component is part of.
    """

    linked_subsystems = LinkedList()
    """
    Subsystems.
    """
    
    
    length = 0.0
    """
    Length.
    """
    
    height = 0.0
    """
    Height.
    """
    
    width = 0.0
    """
    Width.
    """
    
    SUBSYSTEMS = {}
    """
    Dictionary with systems that are available for this component.
    By default each of these subsystems is added to the component.
    """
    
    def __init__(self, name, system, **properties):
        """Constructor.
        
        :param name: Identifier
        :type name: string
        :param system: System
        :type system: :class:`seapy.system.System`
        :param component: Component
        :type component: :class:`seapy.components.component`
        
        """
        super().__init__(name, system, **properties)
        
        
    def __del__(self):
        """Destructor."""
        #subsystems = self.linked_subsystems
        #for subsystem in self.linked_subsystems:
            #logging.info("Destructor %s: Deleting linked subsystem %s", self.name, subsystem)
            #self.system.removeObject(subsystem)
        
        #logging.info("Destructor %s: Deleting reference to linked material %s", self.name, self.material) 
        #self.material.__dict__['linked_component'].remove(self.name)
        #logging.info("Destructor %s: Deleting component from components list", self.name)
        #self.system.components.remove(self.name)
        super().__del__() # Inherit destructor from base class


    def disable(self, subsystems=False):
        """
        Disable this component. Optionally disable components' subsystems.
        
        :param subsystems: Disable subsystems
        :type subsystems: bool
        """
        self._enabled = False
        
        if subsystems:
            for subsystem in self.subsystems:
                subsystem.disable()
    
    def enable(self, subsystems=False):
        """
        Enable this coupling. Optionally enable components' subsystems.
        
        :param subsystems: Enable subsystems
        :type subsystems: bool
        """
        self._enabled = True
        
        if subsystems:
            for subsystem in self.subsystems:
                subsystem.enable()
                
    def _addSubsystems(self):
        """
        Add subsystems to component.
        
        .. note:: Add the mentioned subsystems to the component. 
        This function can only be called after creation of the Component 
        because it needs a weakref to the object given by system.getObject.
        It would be possible to create a weakref 'manually'.
        """
        
        for name, subsystem in self.SUBSYSTEMS.items():
            self._addSubsystem(self.name+'_'+name, subsystem)
    
    def _addSubsystem(self, name, model, **properties):
        """
        Add subsystem to component. 
        This method is called only from :meth:`seapy.components.Component._addSubsystems`, which is called immediately after creation of the component.
        """
        properties['component'] = self.system.getObject(self.name)
        
        obj = model(name, self.system, **properties)
        #obj = model(name, self.system.getObject(self.name), **properties)
        self.system._objects.append(obj)
        #obj = self.system._addObject(name, model, **properties)
        #obj = model(name, self.system.getObject(self.name), **properties)
        obj = self.system.getObject(obj.name)
        if obj:
            """If object is indeed added to the system, then add it to this attribute."""
            name = obj.__class__.__name__
            if name == 'SubsystemLong':
                sort = 'subsystem_long'
            elif name == 'SubsystemBend':
                sort = 'subsystem_bend'
            elif name == 'SubsystemShear':
                sort = 'subsystem_shear'
            setattr(self, sort, obj)
        return 
    
    @property
    def volume(self):
        """
        Volume :math:`V` of the component.
        """
        return self.length * self.width * self.height
    
    @property
    def mass(self):
        """Mass :math:`m` of the component.
        
        :rtype: :func:`float`
        
        .. math:: m = \\rho V 

        """   
        return self.volume * self.material.density
    
