from ..base import Base, SubsystemExcitationLink

import abc
import math
import cmath
import numpy as np

class Excitation(Base):
    """Abstract Base Class for excitations."""
    
    SORT = 'Excitation'

    _DEPENDENCIES = ['subsystem']

    subsystem = SubsystemExcitationLink()
    """
    Subsystem that is being excited by this excitation
    """

    def __init__(self, name, subsystem, **properties):
        """Constructor.
        
        :param name: Identifier
        :type name: string
        :param system: Component
        :type system: :class:`seapy.system.System`
        
        """

        super().__init__(name, subsystem.system, **properties)
        
        self.subsystem = subsystem
        
    def disable(self, subsystem=False):
        """
        Disable this excitation. Optionally disable excitations' subsystem.
        
        :param subsystem: Disable subsystem
        :type subsystem: bool
        """
        self._enabled = False
        
        if subsystem:
            self.subsystem.disable()
    
    def enable(self, subsystem=False):
        """
        Enable this excitation. Optionally enable excitations' subsystem.
        
        :param subsystem: Enable subsystem
        :type subsystem: bool
        """
        self._enabled = True
        
        if subsystem:
            self.subsystem.enable()
    

    @property
    @abc.abstractmethod
    def impedance(self):
        pass

    #@property
    #@abc.abstractmethod
    def resistance(self):
        pass

    @property
    @abc.abstractmethod
    def conductance(self):
        pass


    @property
    @abc.abstractmethod
    def power(self):
        """Input power.
        """

    @property
    def power_level(self):
        """Input power level.
        
        .. math:: L_{P} = 10 \\log_{10}{\\left( \\frac{P}{P_0} \\right)}
        
        """
        return 10.0 * np.log10(self.power / self.system.reference_power)
