import unittest
import seccure

def nist_test_vectors():
    # These test vectors have been derived from
    #     http://point-at-infinity.org/ecc/nisttv
    return {
        'secp192r1/nistp192': [
            ['#',
             '#<%E/4#.v,7<^TdueIy@]IU5S%7,8?',
             'D:/]|&5!rtYmWN,>BDR$OQNC&}Bxl'],
            ['$',
             '.G%F|2_Dl5)AnEw@|/0m/_Q!v/Z1.^',
             '.RGK=RHIy3ewGT-PG.P_z&jS@&TGbD'],
            ['%',
             ')4#*~G=.7Ls:LXwQ.SEf-RWyd*2?He',
             '):%*sl<M3F0JK~OwUp2+5ksp!+*ie8'],
            ['&',
             '$jT2E44Id)YL<CD:B:MWI12+ICD*Hy',
             '+Kq&!<&WRJji0xozZ#fuxe>CL)UG[E'],
            ['(',
             'sF]%z{=H,e7nh!hAn4){e{Hx;DCOu',
             '$UUneyI5=Fup0BP0I>Uq)inTprX=-b'],
            [')',
             '+Py2hL#3v^//Wf(SjzCiz#2W6-!{H3',
             ')x[m~OT_[%gYQIJ<mf{1ZGa_Ms52RA'],
            ['*',
             '*DazVa{_8G-6A;qqj8j1vEhVX1X^M,',
             '&VA2Ap~RIYljG0i@Fwm6hmnPOvDbU8'],
            ['+',
             '$N27AmG]qBaQ%Eo#vdl.H)isf#Y#pa',
             '+OfEK<V!9yd/_K98s/;@;*+qpX}ig/'],
            [',',
             ')gx)i3PopPmdZhH{B|Ep9JvDLt/s^%',
             '+R#Z[xsI9xOLOUExNikbN3V.E#zW8'],
            ['-',
             '+siKUZ.HUi.;hwI|TVJvx(FUs=)PW<',
             '$YY/P%n=~v{ILI!~RMazQh.Wbq:BI!'],
            ['.',
             '#N~?DAEDl%D!PwU3(8~xKq)3|>>?6i',
             '/K4D~^>n-zfyED5)Q.qIvuh(/(MeP5'],
            ['/',
             'qgmrXtieD+Wh,7<&.V!g-iDY4DAq%',
             '&bkU|9#D)(f:9qL$i0=dhk-=(&A1p~'],
            ['0',
             'uJ,/kZ(gk=#*WUrE>8z!98*fT)aq$',
             '(fYT~];|R68nMVYPCIP-0fUSfwCZPu'],
            ['1',
             '#$Ggz#2y&Jxo^QWRH77v>~&lA3V#1]',
             '#YXyt/-Pj_d)H6z=SBx%HJbRGPa9yG'],
            ['2',
             '*?a#[CS2dxE:&78Rxz-VQ4HB~Iq{^p',
             '$!!H^>bjKWo(,{Y,VRdmR-UDr!YVKH'],
            ['3',
             ',S=281l8hjzlj?T_wY5)TLwdHGQOI%',
             '0>[w!!>fC<MLCfG7;)eiQ>D]Rae6Ci'],
            ['4',
             '%T??1?wC<uoNg)em6wCTIW&X_0ET.+',
             '/My]PWgtN~Rxc~R1P#Dr:DYf@l,A_E'],
            ['5',
             '-*T|:*TmbJV}D+dt/?zN#hZQ?<~2sF',
             '#V:m?S!O0?!>L!J@!_YX#MyPhG6smV'],
            ['6',
             '-#CQhmXm}V~_8}E!!1v9]cVHx{@vSt',
             '+1xc4~S3Dvfl.{FUX0P}h=ZxNVX6vq'],
            ['7',
             ',i1z8tG2uj9.Vg(*O}CN25%,vQE/}Z',
             '*XSz9v)sc_1YIXw5Ys.-!tqhGK7!|:'],
            ['=)R$U0!8,',
             ')iZUa%AO~)7Pf_jjlN3oe0hWVI<=AL',
             '/k7BDtG}oUDJumbb~rR_R7f@Gk=&]P'],
            ['h=k])#StiU[0IMl;b,',
             ',A:0c|XASm%!Fi#aZ~O88v<rd^$ORS',
             ',c&Q^mM^|LBN1gk?^jxE|>t^q}*XG_'],
            ['%J6g^PG&e$cqy(ZsU[l$NilC2+7ubr',
             ')+051*qwTf5&{WO=2JC2,|%q#5T4!j',
             '%xJwH{bc8_>])TF3blo,B$}bR<:=[<'],
            ['<RitwKrwCnY-HHONUfavW(q>P?Y?',
             ']Fa.z7VOeXA|4%K([4B@Ye;1%*Gp~',
             '&}czo:Op!f=E.(3r1K?$EsI[Rg]Zf&'],
            ['%@qm2T9*#5MsoI~r;nfCO(~CPRZE&l',
             '$,YlC3H5cJZLJDW$@9<>V>[3TBm8hH',
             '+ZOd#Uk4&cj%5<v&:<vV2(~9N=dkqB'],
            [')_dYB!@[yZ%s}XPvpt.x:La>@F;PF1',
             '%]<jIV;+dI_=/:O/)iavib!EG5ZCD9',
             '/PQ!fSYy>U-y}tZr*rC*n4/e,m97E3'],
            [')_dX3jMhRJ;o{~b<BbMHVO}3v{o.m7',
             '#RYQB6.Fv((}+25.4$Z2<Y?-1Jf><*',
             '%A#T&2>(=)nOi9%NW&aJF[=BBfqhj~'],
            ['&ba4+:I>#QwUJu[a+fU1l%ti(i!C3',
             '/Q;>~77~A^dNQGJdpqptpXThsUErEY',
             '%LBD(p0U:&1o|grzTz#>uOBF51eJX('],
            [')Q2EAWWqv09a+Aq*r%~8k4dD}.',
             '.N-|P2_[S,G=|z?K3,Y@17#Cs3C-,4',
             '.&mDr^|jeayq0-7(S*$lLO~wr>w0OW'],
            [')_ck7)YDM#V)q0B([7#.k9NzFo$ZdP',
             '.>Z7_:3/D>1&EOgq-W#ugFR(R,$Qqg',
             '*t,i5h<~Jn}))dn!xZ(L55:508))j_'],
            ['o-xs/xXptrZPWKFyF!f%VgQbxQd',
             '&R}up{cDIz%pH):5WFgJ07~<abh9=^',
             '#FbT0OLA~T]jN}?CT03ne<*YpScg*f'],
            [')_dW%Bfc<h/0q-Cq]iTB!k!D%=>fhz',
             '%}!fIA[/#PG]GHZr]vQT<}k6&ibNVC',
             '$2XD?dpY(Rccvs*y$LU6AMC,JX,8kk'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcq8',
             ',i1z8tG2uj9.Vg(*O}CN25%,vQE/}Z',
             '(fs:J,s1+V]ll.k1dmT5{<DjJ%SuI)'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcq9',
             '-#CQhmXm}V~_8}E!!1v9]cVHx{@vSt',
             '(1NQO#GqH@)Z*j=mfR1A6sYXBw2bNN'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcq:',
             '-*T|:*TmbJV}D+dt/?zN#hZQ?<~2sF',
             '.i0GDMyS]wm+if:&?#(fzc=#+*T$Wi'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcq;',
             '%T??1?wC<uoNg)em6wCTIW&X_0ET.+',
             'qMV3I50>8;MQg25n_=LdlZlPc_Ufz'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcq<',
             ',S=281l8hjzlj?T_wY5)TLwdHGQOI%',
             '#k=c!]>Iz@yr!=0&XzUKrqt>n&c$V'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcq=',
             '*?a#[CS2dxE:&78Rxz-VQ4HB~Iq{^p',
             '.@Gl%d::A]}B,j+:h/{QK&a/}N1@yw'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcq>',
             '#$Ggz#2y&Jxo^QWRH77v>~&lA3V#1]',
             '.en:msoR#V,@mPh)k?h<TfS~I}+^Kx'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcq?',
             'uJ,/kZ(gk=#*WUrE>8z!98*fT)aq$',
             '*Xm^cDa(;#UWh1*r{925nJ_},VG<tJ'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcq@',
             'qgmrXtieD+Wh,7<&.V!g-iDY4DAq%',
             '+][]eix_f2*/{t8AURDZ6F-7kJIgTA'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcqA',
             '#N~?DAEDl%D!PwU3(8~xKq)3|>>?6i',
             't6ocC]6a<)LpCO=mSnu(;NLdI=3u.'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcqB',
             '+siKUZ.HUi.;hwI|TVJvx(FUs=)PW<',
             '-en)3|.fl?p|i>bDl4~DKI+y0]PT|@'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcqC',
             ')gx)i3PopPmdZhH{B|Ep9JvDLt/s^%',
             '07u4)E#1C|svi8.}F3vS<c%ze,gzn+'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcqD',
             '$N27AmG]qBaQ%Eo#vdl.H)isf#Y#pa',
             '&obo8fE$S=,:U<K.KSF~d+._~tk/_4'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcqE',
             '*DazVa{_8G-6A;qqj8j1vEhVX1X^M,',
             '+i*%B1zPC[![nVy%whs,6CH#AWF6p+'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcqF',
             '+Py2hL#3v^//Wf(SjzCiz#2W6-!{H3',
             ')FkFcQFD24(ld>:*Pye1BiTrCZUfr~'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcqG',
             'sF]%z{=H,e7nh!hAn4){e{Hx;DCOu',
             '-iqE|)QoOovV)E46uD,MuGG{~[2Z;^'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcqH',
             '$jT2E44Id)YL<CD:B:MWI12+ICD*Hy',
             '&sV1bfuK:l$^(lrHd^yI%Kx1EH5Oiz'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcqI',
             ')4#*~G=.7Ls:LXwQ.SEf-RWyd*2?He',
             '*)E-n6_UYp^{ig4KhpP7iEBbpFb/b+'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcqJ',
             '.G%F|2_Dl5)AnEw@|/0m/_Q!v/Z1.^',
             '#m!iFNRXr&*Nn3VrwT1_$-K}QJ6Od{'],
            ['0@H5c!z$l7mG6fc6P500Pfc67rPcqK',
             '#<%E/4#.v,7<^TdueIy@]IU5S%7,8?',
             '/{1)&%uokCwlH06;#@=lyae%NIkTLS']],
            'secp224r1/nistp224': [
            ['#',
             ')xnVviC:vn|W;:.Sv[|,{&S?w&jm|Z3;nWX',
             '*1n@5yIW1#M2/C!,&$Oyi{tXl$;^Fh0,UAm'],
            ['$',
             '&:+-p!IuDm/>fU|:ek?KKo(z83N::@Gl[jq',
             '#(~jAx%9-X){!1V;@PIwF$xe|z/_Ov@7%Ol'],
            ['%',
             '+K>D@oaZxra];/jnr/6>hD:Zy~IOr(1:2zQ',
             ')5jef&A5$q9*_q,YdPS<N(Z*+#V43HYe-[0'],
            ['&',
             ')Z!vm!8]vP)PDBS{XIDZeL!zCZrQee<YiI8',
             '2C|_9[22CIKZ4BHrNfe7Qt%5}Fb{Gl<~p2'],
            ['(',
             '#tY#UH^0H*EZ_zo=txd<Nj>Zy4$d9ZtM_B;',
             '#Q!<dky8dYQx-Ihd.RPE!5xkopR3?iwLj0r'],
            [')',
             '#3.U./e2X~4q?Wxiz17Jvp;@0eL$8*IQ8Q[',
             '(76$4k{l*:R=L&9Q9u!V0iNL30hlpEa@M7k'],
            ['*',
             '+=yt]8s#O$S9As5,l}wR/x.^~Okw7i%5{IW',
             'Vu:rJB=1wvQfW&(|@XAkt_pCb~oE5?rBgp'],
            ['+',
             '((/6D.WTk#&F4H3hSZ1*rF3Snh-MKJ0v0h9',
             '2+Cc+-:sPpII^cQYxw4H@S&CGlY5M;{JO{'],
            [',',
             '#n,Xp75[;qBJvWd:<Zm?#aio(-Qup_Nc+xd',
             '$*gNqJp^F2VTEsXx>$~Ss+,_n3,V7Fk@i9t'],
            ['-',
             ')Z6yYS93O8~b?[h01/_f3BV,Z_YU=v^i>C4',
             '$3gD$9d,hbHC5-f/svB%9lWawGZ{>jhK0st'],
            ['.',
             ',(WI|1Cfx^aSNuDVUFo?xcDL2+T5YdzKr[+',
             '#8LJ<msc)W<h?nJ>OJq1kOHNh-H@Sh2:^r['],
            ['/',
             '&2R5XW-8fIK7{g:tzIr3T(XP?Vwh@d|lbuA',
             '#7f>m(,OI]#m.ol#<o82xJ*;5+W_n~h!j!t'],
            ['0',
             '$!>Wzepz|=>dY}vU7],c|7py2#0c2^7>Sck',
             '#G[)b)nn>1u}g0,]UbftdF7@it]w=5gcV=*'],
            ['1',
             '):#*/s=Kq#m?u40qaruleDe<yz/~~n%MIb3',
             '+*WiA^?>B>b<sK~zfpO~Cn0Cx4GOXkv=Am*'],
            ['2',
             '*)&BKgofabU@D_pgfc1AcZv;-hFUr:^b-Q@',
             '(g]3+jZqZs>dEC+{R40]@S[U]rK*hhUqu?v'],
            ['3',
             'I{K:3q;G2q[8<+Upcju4-uSzDn_>-|Z{/8',
             '#zq2w0%-g8H/otU!3fv?U$vQDQcKqFC0W>q'],
            ['4',
             ')|hRDrDq,0$l6xjIvsfYMqvK}z%kS[v=vk?',
             ',_;,;=7]F?rO+2(iP&Ftp,C4($-FJg,,Z+E'],
            ['5',
             '*^DD@GhQ&y%3=#9?Xuwoyw]s6I5^rlo;5rU',
             '+tLfjdVC*h9Sx?E~F4s[{Q.fmU}j0[3)0SA'],
            ['6',
             ').hB@$S-_i}}P{,|VXgj|_D%BTU<[.pvy@$',
             '+Cy_W;]IRQ>JO8k?p+)_+pj>MKkXzZ.sk$f'],
            ['7',
             ',VIpAcCrWd<S9WXvFkLW}bD],^}64E5h<]R',
             'PT+PfSm+/ZVpP!WO5sw9jFs1OFXYKHZ6y7'],
            ['=)R$U0!8,',
             '%diA=Q||osP!<!C((JmBs1gfZ=X1[4WjKM5',
             '#>Fa/D><%.a}*U:Z:v<vF30rp]AJ%ugF[b^'],
            ['h=k])#StiU[0IMl;b,',
             '+q6:-r.QnD}H@<W!l]8MAT)oxH(1B*s!KH',
             '$DPV7[cvKNlU7jO5avzWUzWiUuB*iyDN|8P'],
            ['$Ow/lw2{%|I07NU8:O]c3hV=QCAclF4AlUu',
             ')O.ZTqrAbydk%KCD*HiOvn@FDcup]wB9Zw,',
             '#yf=}VbCFCT-XOYeb&5,GFg{A3CfizxUL<]'],
            ['&q5&Koo%#%M4pd5n0R&w@rV#o=L2E;hb9;d',
             '*3Pm+K,E/+M,uEGn<TbhJ_=nCdBT/$y]j&D',
             '$uNmCXq#5/O[pPHfNs{!E&R&I46:|2>0o:M'],
            ['&q5&/{&LWEq>n-1.f]3rP*Ug}{8t0w^ZWf@',
             '+*epC|++8w*nN#Zjg0KZB:E$15@L=YKRe2p',
             '*{^D^+BALVbsEy%Y0sktLXtr_JnqBPH>ig4'],
            ['&q5&KlSs<+RkPNB8|Kaulbg2AY^B^J,n*X,',
             '+Bq&OdY1N,mrc+09U_U;Z^IY4T~l!7%Op_L',
             ',!a?nW}V{1/7G(hpQbW;$Q;Cebo)6K!|~fr'],
            ['&8d-G))Lc#Uu.F(fQ,S*8)uv139]=so0ZDg',
             '$ks,AV;(Q#N>&oy=m>+:th(9)3Cmo(x{L1o',
             '*9p}tp+)DB9Sfa$zFalW?b6s/,GS^f}[%_Q'],
            ['#_qn-=gwi$qiOfTe6x?/0.7KPCN<yA',
             '&x{:tg*L*Zn]2b,4g>lmJ62e]c@93Uy4^8j',
             '+kn!6=mWX6Qn+i]I%!:dFJ5IK7%=pd-A2x0'],
            ['&q4K^.>hKb>llLsVc}PR(VxgQ}5JC2*&K@b',
             '%t+qP)Ib4|v4Vv*n}J3p1Z7CZvG,02rqGV@',
             ',3GZ?[d0q5S_yX<-V>BIg(Vd&r;R_TpG0Xs'],
            ['Xw0+uL@}ofSoBI(%YlO#7i^y<q4qn}w%',
             '4et2U$3cfxARW!ssRKF,Nb4Ism?+rcPe:',
             '+(#9PjNVNo/5#E;qrU:O&|zoK}GS{14lLL['],
            ['&q5$t8$[+9xc87Eb{s@G:EN0@|#r)]>pxgZ',
             ',(2wg+mjehE8rAQ4WoglR|5W!oFnIKJ/iS0',
             '#;%w?K^Z1;OMSA]d[jG8QUlv]o#DMsA#(L]'],
            ['#3Y0)*g<}-^a)fT7{ZyrL?]sf=fLiwG',
             '$Inh$]K|RL+P?V2?[Sp<^edaxs(4Lx>sonb',
             '+1({EvtO=rGViNjIR5AZwCiO1TnjQ8y1y.z'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xl',
             ',VIpAcCrWd<S9WXvFkLW}bD],^}64E5h<]R',
             ',3r!Yn;Nft{VH:a?[>kG4$U=o$>]o]&4Lv.'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xm',
             ').hB@$S-_i}}P{,|VXgj|_D%BTU<[.pvy@$',
             '#@LJS;2r=Q;diQuW<IW_A{3rQ(w]NM@xvm['],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xn',
             '*^DD@GhQ&y%3=#9?Xuwoyw]s6I5^rlo;5rU',
             'myDAp8xg<@Y@K<vf?kbN<oJ2zeM<L<gS>#'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xo',
             ')|hRDrDq,0$l6xjIvsfYMqvK}z%kS[v=vk?',
             '%.~q9W^Ide_1XZ/ZM:IYcX|xOWp~BCd)g|'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xp',
             'I{K:3q;G2q[8<+Upcju4-uSzDn_>-|Z{/8',
             '*gTx5Fk2*l2!Ht-uxkh~tj&^Z!!kWc,_,RP'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xq',
             '*)&BKgofabU@D_pgfc1AcZv;-hFUr:^b-Q@',
             '%zix#j4J51;JsGVyX@Pb-:@YA_:0b@uzlQK'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xr',
             '):#*/s=Kq#m?u40qaruleDe<yz/~~n%MIb3',
             '#YnAjuO}MeurE>azEb1@)}mm(@=gp=UQB%;'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xs',
             '$!>Wzepz|=>dY}vU7],c|7py2#0c2^7>Sck',
             '+<k$JL~MQrb1QZV:ToxIgGfp6](@/sf--T;'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xt',
             '&2R5XW-8fIK7{g:tzIr3T(XP?Vwh@d|lbuA',
             '+Lal?NdlFFUB-xttocI/QCsukI-VZ*elwoM'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xu',
             ',(WI|1Cfx^aSNuDVUFo?xcDL2+T5YdzKr[+',
             '+Ky_ofyXhK=Fxz8X[)n0_>Sb8G<vuA=T$|f'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xv',
             ')Z6yYS93O8~b?[h01/_f3BV,Z_YU=v^i>C4',
             '*P_g+=-3)B1l&]zi8Z?<4!DO)-*<.>eCR{M'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xw',
             '#n,Xp75[;qBJvWd:<Zm?#aio(-Qup_Nc+xd',
             '*Y_Z;+|]Ir!Xru)|nN_jVdqP2AXa5bbMxWM'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xx',
             '((/6D.WTk#&F4H3hSZ1*rF3Snh-MKJ0v0h9',
             ',R>gIKd#zQgeo,}EPWh.$MI-[,v]7Z3q9AF'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xy',
             '+=yt]8s#O$S9As5,l}wR/x.^~Okw7i%5{IW',
             ',-Pp:,L~_,a[R3[p04(|^w=@ZodH&t/zA+Q'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Xz',
             '#3.U./e2X~4q?Wxiz17Jvp;@0eL$8*IQ8Q[',
             '&M4)whqOgj&qleIEr[^h=%MdmCzJXclN6YV'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/X{',
             '#tY#UH^0H*EZ_zo=txd<Nj>Zy4$d9ZtM_B;',
             '+3FnGht&-I(7/@x4}~0yJX$E0b3(-?TAwbO'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/X|',
             ')Z!vm!8]vP)PDBS{XIDZeL!zCZrQee<YiI8',
             ',R%.L=4-^a0c]V@N:$xY6<*.jS>TMaaQc!3'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/X}',
             '+K>D@oaZxra];/jnr/6>hD:Zy~IOr(1:2zQ',
             '%N[EFON*l3A(WwV=H#.${gB*uP/&9_r+V65'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/X~',
             '&:+-p!IuDm/>fU|:ek?KKo(z83N::@Gl[jq',
             '+[G@jZk%dJP49Y,[l#7G&j$K!VUVy3/W^AU'],
            [',cG,-Sn<o$W09iav-Nnu*-^|e#d7k6jO/Y!',
             ')xnVviC:vn|W;:.Sv[|,{&S?w&jm|Z3;nWX',
             '$RWjvYEda#,}-Gam)O1Dap)V4OIX%A?d.OT']],
            'secp256r1/nistp256': [
            ['#',
             '$zKZLI[$GRJkeQ_|_uM{:Nvg@r|TRr0p]P)ukU%3',
             '$5#W5w%+yATVgas6+(i7[8n4KAu/rCA*C.A,c}Rt'],
            ['$',
             '%JdV1Ct4}9KQ9A,/.MGBfBeL%C@rByHRz|8s<g>-',
             '5PV+tLJ7]*e[Ew@H/SiC{h<!4Ka56ne+Qq:0xRJ'],
            ['%',
             '$Y|L@fDog=5xfNC&sQHMK)j(iz,VPQn&-c1;y@3}',
             '%f.ooo4N9P9jghm^]?lT[Mf:^73YBv._CZ+#|h,W'],
            ['&',
             ')7~=9h]R86mPdb0MC,9J71,r^q#*>@-%E?KrF@_[',
             ')4UqxYfrv.B2nB85NVxz.,AQ[=C.I0/<a^8Me!^E'],
            ['(',
             '$8]>/@?ZJ/@=;F2+nWppawODqR(10b+RZGv.Si?X',
             ')4,1/koiBEU$XBJnx2#MX&(Hqw)>b1q@9#^M,-,-'],
            [')',
             '&qP>aO(&%N*ligxF]8m~n#iq4%kkP*NH2KYT|+EP',
             ')FzJ-L!WCtg)litRREnC#&6Ph8Gxf,B(y~CQ+]_e'],
            ['*',
             '%wk<mAbeQ3g}GlX)RDu]TO%~D#5!L!(Pq|t4R4hL',
             '%4@GPAf_dp)n-04AeZNW)&s}Ty75ba1h+5eia~>5'],
            ['+',
             '$f#Za!D~Q_y^;$xQ/0;K<pq5+ValOm:h1zV?^evL',
             '&jcK>_C|X1tV4k3SW/BV-XFGixH;9?dOcMO<q*0!'],
            [',',
             ')L$g4RQ0X@->^&zasMlE-a0(_/HBt-N}S#5gMnDw',
             '#1Ncdq?h:7Gf-hw3awZG22ipNs*1QC!c$Mxfq/i5'],
            ['-',
             '(c~ld$8QjoEx?Hr%R&Xema=_}opTUn:;dw|K:n@v',
             '%fx<L,:$mFkP20o!pjqP+w-;d/7>Ukh=#bj7PmyF'],
            ['.',
             '#fd@E{oO7Cs|>ivEnDRCB|U~$+{RCaXRMZPDX3JF',
             '%}HCHAOg#djH6N]O4so;P*wHhPXi[jKj5bUYJ~XE'],
            ['/',
             '%4nY9TZC7Ca3&,/8<#XvUQP/>CXx_z;kI]M2TRiK',
             '5Jl?TpvpW/%&pw]oVWVZikiA5BpK|R2&~YH+Yb]'],
            ['0',
             '^6Gb/4q6&yz3p-&06[^(:Y!Z!-)(OchO]6?*0Bt',
             '$h5Kq^T)mR4xW*WSUO65G7b$|AKZ&h;WdBAdT?t-'],
            ['1',
             '$AKa(XZgoIi.V=CC6ee2u^>I6RiM0uH++=$v^vaR',
             ')ijJ*<Ibb.GTwF-3OBtV][kBl;Gv:H^Dc_y2O4,|'],
            ['2',
             ')ZU:b;cJ.mFq-bQr2,n%>8afSZ4/^,OsGHceoN4n',
             '(!JH*&,sv[yTu9m4wlP*gwRMIjkmw&vnH>@4NK_S'],
            ['3',
             '%;.>/ggDgNlmV8hh^m?n*?N|by:.5bHUoSw)H@}$',
             '&a4i+F&Mfe4z.(=X*cc+XgPt<#p.TP=?8!Ye|URM'],
            ['4',
             '#|/y,O]Y=/QItv%7|I}@30ZqE{p6%VHfTak?/NM1',
             '&bEFo/ZgdQ@jX|Q1uQTD^xeIGGIrQp9uoaEp,]>$'],
            ['5',
             'KPv9HV?,*&E&QmTlQ/k(OR8gBamM6=x&FGYmt=^',
             ')m+epy9~Nb1wpL)sX*7FX>J2F$:W2sMh@~D7*C6a'],
            ['6',
             '(Y7xZxpItlhE&=|seu|h4p3O)x~?IStu<v$DcPnj',
             '$K5|L8EtH^L/|o!_AFZ&Li<ct%RRt5v2B3E[Bhfq'],
            ['7',
             '%[:Cc9B}f6BA2U/3_STk=9:ve7bjsi-*7Lv:N%Be',
             '%;c}DCta.{Eze8r{ZUk)Pf=$DtHdK.;64W>^|og&'],
            ['=)R$U0!8,',
             '#Hnqbg<JLd,qXoE*/t%>PpMcA/&Qc1M-}w..L5*]',
             '&uY,<U!/&#eGm7Kt*%~>RS1|-gX!sIJ:8=MH[7SD'],
            ['h=k])#StiU[0IMl;b,',
             'i0&H/dQXh=WK}uMWXX-e>,Fk&bRS*h)N9Z_l3&7',
             '(<d/VXdMJbwHfAugvb9(goA</pn7Lq)-rVl@?Qd0'],
            ['#nX9m:Z|$7teS}?]@fTI>=(-Wv>}X*/0DQu]si>X',
             '&D>i8H+?VVI_]JGEplJLYw+Wzim=Rd/2:E7*$u6#',
             'iiRS+PFluXQ+xI%v$[e^]I:2!)FAV,OezHf|u#{'],
            ['%R;Xe*59<7(kU(z6Xa+*E-3ykGfE*6G((NyWL4zv',
             '%g!vhB45oRSrCH9k?5sR;W*7YVU|y{1rvXUGsQ5v',
             '%-n1=K9;VgFC6j}=uUJ9{1(O2m|OY!.iG2TBbEfi'],
            ['sIMQ(}(NF/?95s2fyi5TSDhWW/y[({@{OK@/V',
             '$lphM<VYlEe=mlzvM*M*)1)p*d}Cg[vU*HlWJN=E',
             '$%%COO([B|mkxj03@FhKoSI?!T{QVE7.O6l:1+r&'],
            ['#i]>&:C_b~v}~5kix*F(c8m}&C6b1B&blQfl/tFH',
             '(Z2v!O5/ps4El(y;(QA~fYs9*2gRifS[Ui]!KYWG',
             '$*~/ws[1vNmE=Z[Zz5/g^GX:E50-AX->6/{yP9[M'],
            ['ZjC^[)=j#jpyASa/8JJ)Kc]qwd7W*{g+0S}jay',
             ')[l6;[9m)cPh%>]{}g>SlqRdOcqVH$0QPg0!2X*g',
             '$Q-D&~%8~@2kV<qY_aY,lOh5O.yWdhvA<}*]}=0]'],
            ['S9ZdEM}mors!LL;(Cx3)hS+e<=[0.:M%P^R=',
             '$Y&BG8.vq2]1P~^3me83JlxdgH;6JH*J40nQw/}?',
             '#tl*q5HS-wGzN4MXhw^{qm8H{WDmj?eQgPaOC;9%'],
            ['%Q^oW)aVW.coutN^)GlrzANC[p;98@sV+eq0RKKJ',
             '+Yq<8T+>[$v:kCh{[p*x6l)_mR8;,~U.%rS=UY3',
             ')zknWub#b=!$Y^J7?O8lj3Hv6[:z4BWG2nvnb$L^'],
            ['%R;WV*w|{1iNpt96<Y7,K].]+;e=#d%HMI_O]&|w',
             '|y5nG$)8SE/).io#?doP4LnBjSKEk|AE8ZQ$z[&',
             ')qLl2QqB#y:Vrd^FNjY#hu6~wprqmO8Azku>sy6h'],
            ['$N*K[pZ!yI9!BoA&OjRcu{uKJ41ClEhb!-&vc:,',
             '(AXbn&PodWyMEZ164oGIGW0L@2Y,f]l(1f8yS:Fw',
             '#Wv3iZOMmHTfRg!haIK+GeI?NE}_ab0(fA|OoA?>'],
            ['&{/O,Z}!.4=sVV]vw*R?jtj||0?sfhpp2]Q-.y;',
             '#L(;I;mJ.W&p9>UaB_9fU^Rw1W:VB~r2CE<mtr7B',
             '(e2AB[7C/b5wP*}j[C#r{T),TV>eSSc_%b|,{v3R'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSb_',
             '%[:Cc9B}f6BA2U/3_STk=9:ve7bjsi-*7Lv:N%Be',
             '%jq65HfVpdLJ4OlH,cEEn%sN>Hz!C97s9qvWKP/>'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSba',
             '(Y7xZxpItlhE&=|seu|h4p3O)x~?IStu<v$DcPnj',
             '&YA7-T7CU#F8xw^eErTGr!smm:p2x1Xw,9oZ)W/P'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbb',
             'KPv9HV?,*&E&QmTlQ/k(OR8gBamM6=x&FGYmt=^',
             '9KMfqC9N}bM)<WP/2y(fLfA=<,.[P$A-Jq#A|^b'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbc',
             '#|/y,O]Y=/QItv%7|I}@30ZqE{p6%VHfTak?/NM1',
             '$D1lh]~P;0QZ>j05ogZ*_pK*;uxq<S93ZioG?cV@'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbd',
             '%;.>/ggDgNlmV8hh^m?n*?N|by:.5bHUoSw)H@}$',
             '$EBJNFUj8z^JkbCk]UMCf$^[G<RW9s5j6IYQKjAt'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbe',
             ')ZU:b;cJ.mFq-bQr2,n%>8afSZ4/^,OsGHceoN4n',
             '$&,kOgPD)$vp#Nr2mL_DVq]%9RVuu?X;%.u%yt5n'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbf',
             '$AKa(XZgoIi.V=CC6ee2u^>I6RiM0uH++=$v^vaR',
             '<jiOP3U=SJo~BT47v;tb/E0u$zmSzqdhj<(y/iE'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbg',
             '^6Gb/4q6&yz3p-&06[^(:Y!Z!-)(OchO]6?*0Bt',
             '&>Agf/)22/^L@_)p1iz9wSNMd{w+h[7Oh*sRt!~8'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbh',
             '%4nY9TZC7Ca3&,/8<#XvUQP/>CXx_z;kI]M2TRiK',
             ')q,G:7jA/*eBqugfubVtd!DhB,!sBF|wFJYo@f3e'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbi',
             '#fd@E{oO7Cs|>ivEnDRCB|U~$+{RCaXRMZPDX3JF',
             '%).p1K-Pz{)|c:#tREA3nb9*xliz2Y&?8h^[}A;|'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbj',
             '(c~ld$8QjoEx?Hr%R&Xema=_}opTUn:;dw|K:n@v',
             '%?Zw-bC62;(tgWpCtN>{6r&7}2/G8WhlIhK!wQx{'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbk',
             ')L$g4RQ0X@->^&zasMlE-a0(_/HBt-N}S#5gMnDw',
             '(u(Ory=OeJJ_k}h4&AT(0XFa4J<T=#OFH{<PW4-0'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbl',
             '$f#Za!D~Q_y^;$xQ/0;K<pq5+ValOm:h1zV?^evL',
             '$;qh;.9;EO{nd{Mp0-mu52j+wDzJU&lWh{ezW9fB'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbm',
             '%wk<mAbeQ3g}GlX)RDu]TO%~D#5!L!(Pq|t4R4hL',
             '%r6l)JtW:ojVlXM%!]at9e<R.D/P-dAAC7OMhAV0'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbn',
             '&qP>aO(&%N*ligxF]8m~n#iq4%kkP*NH2KYT|+EP',
             '^XiL@Y_Yk-?,|jq4sB,>ey~y(zk*;1#PJqf@c5]'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbo',
             '$8]>/@?ZJ/@=;F2+nWppawODqR(10b+RZGv.Si?X',
             'rK%I~kN[<=C?F6Tm+/}ff,*oF=G.5^i5HUj@6j8'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbp',
             ')7~=9h]R86mPdb0MC,9J71,r^q#*>@-%E?KrF@_[',
             'q}A^2tE)SP6+FI28b7Q4_n~(#!WE7Cljk|ie@6|'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbq',
             '$Y|L@fDog=5xfNC&sQHMK)j(iz,VPQn&-c1;y@3}',
             '%@HCg{Hif1XZ1}qf*yCvc=J9%*3,KMDI*o.6KWij'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbr',
             '%JdV1Ct4}9KQ9A,/.MGBfBeL%C@rByHRz|8s<g>-',
             ')q%[Mv0mh$ia<Bh&?-ZczmH7b,w%Y0bDBwC};GAw'],
            ['*&T4VkZ8|_qEvg_Df:1L91z@|8h@+%NsnSWckSbs',
             '$zKZLI[$GRJkeQ_|_uM{:Nvg@r|TRr0p]P)ukU%3',
             '&qRZCsW0%@=n2(l1]4G7cRB?7{MUz#2!+>t.fBAM']],
            'secp384r1/nistp384': [
            ['#',
             '#?>g,R[~H^zenwA+.lO&EvJ.AU0X%mO&n4,j#y(2xT1.BeDn@9c<V0FG>mfH',
             'HRR&noUlCUO{vD]C/Jjr_ZK7TY.<=[L-cqIULK]gy&:pS7Zn|.DG}Y[E-,b'],
            ['$',
             ')/WS~1MF6%Z+c4,s}LZV8Ph<0_s@^[r{v/BNXR}Ih,@H#K-dYrrab|FL,Uv',
             '#+rjmCG?0&n(I[Vb}Wx4kPOp@UqOb|sT9M7[A8G]#4gi@U+Zt*|$ScVg:E:-'],
            ['%',
             '(3u5@Z|2?_[QsnW.5*TX>LMd/byEjh5sK;[a*BZ-zQwv3R(!ndz5KOPN|$]',
             '#Tqlv>mwUgPFVqkiBH-z@eu<6)X*v!.%q|X}@zhF)0ev*=JV:sUb*bbjby6_'],
            ['&',
             '0S.5hvE;?q&:xy9w>/,g25JW2fGbTNSOc~tg-b}:78*Q=5OG!yr#UlaV@Mj',
             '#Uc0!)^_H>xq@}P~NBqKLO0_10!M11#e3@O=2J;?e@!{_-=ifw}ykD$*Ih*H'],
            ['(',
             '/F]pWSV#h*D|kX:x@%Z~HfwA6<1}0F5@Jl*P$F|_xn29[GSwMGHAP2~,b-/',
             '#,_F[k{Z(6452OhM*DP2<A@{X:L!PUI(54oAaZRDM,m]]Ow=Es=i5c>,U.dm'],
            [')',
             'i@7U%5OR;W)r:(mS@~b8_rLZ46fP!WK0p^9}E@ijJ-kf(mB5C];k#>eLHi_',
             ')UosYO>NiWB[fhBe#3D}tpHVflR%k&uNXSX,Pcj_B38i=^PA(zT]y?.)9~U'],
            ['*',
             '>w6bCKpDx=h03EnCL>WAk8H_p?e?{7<Ziy}ShI6cfR&D(LSfr8Z.43loe<|',
             '#/~bt{p~Fwph=go=yNZuZ?v=Zu7hik)z}%or.US,e1H9aNnw_5;cJQKD-,+}'],
            ['+',
             '2_E2Feuu@rA:a:pu.!A>ts<IV5$:yvSPJfpC]e,oJ,q6L5;(2JkPa]|WGkD',
             '#d4ot(LkEMZ}YVFlg=c4O+}b68j%RWcrQj.ZkPcb]L(}~S%e~71~#Vs!Q7kY'],
            [',',
             '#,8::KGZBoTjBW,N{r.u.aib7V{3-v+1_Ap:sq;>=s-Y6Y4ecs@6V(eac8+:',
             'iRa>PGgTh8gx76XC+6kR[E|,R#<Y;M.87~7g9!2;.uuQI9fpgDgqWYfDz_s'],
            ['-',
             '#<K_0SF{C:CFE)ARrlU)X<}W58eOiq/a19Nb/$WB&agqnTh80@F9Rqn,BX9h',
             '#>ZQxUU&~+nWPxD.v<O~Y8csp>cUn4|/@ZX$pPeaU{nydS_j/jNO4ey8xrq,'],
            ['.',
             ')]0ySL)C)m$}+<3eY|XDYfSf^?czef1m5)4k;Sv6I:*bi^pxF#;QojKK5?;',
             'C-gIYrC_Gz0gr:r1S]vZ?.C:<g:pR_s^TL$XdND]cob?ugrfs.r#8TE74,0'],
            ['/',
             '#0Ni9:o[Z1Nu$J269S#o:[uXi_PUs+9o.d7&y|T+3;f0Z%@-A^pUp/Rb=PCs',
             '#8/J94ouT}Y?y,iAT{on0C35]|.qxqTT-qO{&yIg{r/@Fa5Vc=;+A15T6WAo'],
            ['0',
             '#;heo6SysC^V5N=l+54[R|tM-z}Bh5PAK.hpM^#Eo+g?O;N%D>5?)M=a,[00',
             '#e+*#f~faGCUm:])1YTDHC,$VlPpsy*ohY]T[Jv_}4M0Jw7g{jHdKBs3.dyS'],
            ['1',
             '#lOq,sb)}o4io~h;}iUBa.)kX3:ghBOHH!#4}EkeYT*SqFAjSV(/hQKYa=S}',
             '#(-c+etEj{v;k^X1nXRjPu*3+AmND~r!w3D4OI,CM>u=:Wa/1yPtx:8xY9p5'],
            ['2',
             '#EhY0J-T!KSQ1/!z]?p3wmSWSUdS^{o8{u%)bY*_RB=E(h!HvU2)GujiRKu6',
             '1b8io&:m42j!WzT6g!2Ae>&>GL2FI9RDMjT-7Mx:y0DP=h696@z^s,!==g{'],
            ['3',
             '#^C?ma>A9[5LLsWWn5B3DWV8.flwpVcsJGA.O-sTic=H-AC:9q1t&N80Sk>n',
             '#_v{GPUQFPFEc^a5E-Fr3AM=RZNG77aIFwv_paLB!l#R%^+1h;ao;Ml2p{Hv'],
            ['4',
             'OnTK|}7s%8x+PzIyp>VXS}i@JRy]D_FIi<utJzZ#)H6ZJ+%~X>ZFHK4.~_]',
             'fjtoBqPY5h!S#*|^uJfJbGm(4^*Cytn8!cS4EgzD*aG-Uo.5ySaObUNR0]^'],
            ['5',
             '#f4=/5pEsv.A3~)84&ShQFPty3:|AQa;mRy1kD&fPk&_?m/TgFf8JlvIKW?X',
             'S$[/to}:5n&D;JwH6VL+?:^R%okl@eL{9XXM{([4v^9~zg:l=+ut5?oF>$-'],
            ['6',
             '#+#MolSLbcCH/qqnD<lsDp4hzkUx8S83/VxC2[4I)Xw]r}A,SNC4d[/2UI}6',
             '#8yHYxe{sgWv-D=Dz*mw&FVPwDq[h.B;v[5N8sqVaS<:Wz|b!v{tL{&?o:k@'],
            ['7',
             'gosntT4)N}VtGtjB]B4%*~<fu5.yVwoqX~.J3NTB]3g>3^2&CS.XF8${q|,',
             '[J8ogr2W0Feb&q$YP&{k{8n9nNet~b[3nAd!eQymaZ.FKADe6orDq?L;B6%'],
            ['=)R$U0!8,',
             '#;4RK_e,UNYLp.xj.uxCCXOu%E!>D.U*U@GGCOsE!a3_&88{5h1cPZ#-JQhg',
             'L;=Q%^,Moo5Vwpo&}]lrHOL&9Sf#Ql|[iht*t3aK*p?KcSM.N*,7a3_H+{9'],
            ['h=k])#StiU[0IMl;b,',
             '#-AFgmk/.Il.[4fISD7O?a}Lio[+/pYUyYE;k=8~b?>tpMbo~>F:B%O98z;}',
             'EGSF<hy^wHHM!N@RNJ]X[TV:=N&PWt{l99yLEJVi9O|.^rYy?LFNDKvlu.h'],
            ['Pkvo9n(f5UpIb$VQ2|dDjcg[SL3mU@W{Pc/3i&^U]UclrHU4uoP[nGF^mS!',
             '#s@AQOddQn&r=:sxWbbh1<OY@Xvk0pPI}Kb7ckL22Cu5#<tZC~+,aNO*cH%J',
             '#_,gw:~[Rt1=*wV#wX~ld@=^}w|}V(h@nO>ImGM[@Z1+pZP*F$u_CpF#JRGc'],
            ['OG+/AhyOvMJz_yODX|yI77rGa<]acz#1X><?-hXgYT)($q&AK~yE18KK02(',
             'e2P&pql#Ih)Tj?z.?pb=47e+FWI+Tm&tG2RP_Cj,1%HwkA6A95ZqywdZ3*q',
             'sR3GRj<?TLRv.8pTX,.&B:uZI;StJ9P^ltdh?QG!c)%m<a(J97fQc<cArs8'],
            ['OG+34K2LM|&>v&(RZrLt#s31{rr2[my4tH[.MX:mjOBvJ!-H;KM.[0<.s{i',
             '#dUQUW3NTg8MjvF-b;V5{ugtPLf05>~qy((3iq}d>&yHP0L;!&B0!3E4l=((',
             'FN8^MN]aV^6c~]Q0^:~(=ZjZEcz7:Ca<^*bcUZcrbrJr^J2Y|]-/m/,,|e+'],
            ['AP^vHy&1Ync#)uwt{x$UX4WbG+So)?vMxA{q~CY0IR.]&qc[t$#+3;',
             '#V6yVV,Q-Z~iW0ir|-lQ-i~NSGKCKg8k&$|rZi;nrzRo)6A>VQ2R3r?CM45W',
             'Q*&FU]Y/XQ}%,+ac$Bshf,q)Eq0xuh-a(c5@T%2WoTHt_4C.g%g3.s/k0vf'],
            ['%v5hOg,mwwL=~.+!PLO4-m^G3M=zW@+AR_(d@ArI*wOQZ<64U$BX:aXm);',
             'm.MD|.I8|[LiXjP:EJ^biXnnF?Z~Ju3bVX%Vv3UF$WxAHT:n8BM/hMcC0rE',
             '#-QXdl8q7A/n[GB[jf~pH3gzVF3!wqX|7L/aRJ+cTM>_wB_UC%[<IG]NOcaU'],
            ['(TPq,cF-VL@}Y3oZ,:ESyYaW<M(cE!VY*PDR2KyyJwb5TyCKz;403CcL=?Z',
             '_I4)c4t>XQTGT6/wpgsBvH8nuKonK}.J0l%xV)@/Ld(~vEo]r=i2Lo:R+Pz',
             '#IXDc32~^^|^^m$dDu;-fpj[*tZY(l_(L<)R$2:r9r)5B/d~>m<~#K=Z&7+/'],
            ['#x[vX3t}*DK8aK>X[Ds~||qago*>I}K?Tj&BF<YJcI@q7;PwZ;hS**8L',
             '#0RLU]_YGvHyiQR<Z%uGb-ec7ZjVn@*WjS6U1*toU[dz>TRG}>jV}TO=0gVI',
             '#=28VS+x)+Z}LJR_n;6O1/=(yI|XnA4XeC|$ILZ^Xac5U#gWa%1F3sD[ck{q'],
            ['cJP1[}n;TrX9F^&uNIe6^K~UB][FCy/c:>-L$,KPKB7fP]%(<WwqYy0MFg',
             '#/1)>M(&nhE=c6G/oXEigduk-k#=YHsd$C;b9YB!thb}PS#|>dh5=ONSJ5Ru',
             '#smlypNz/+f,_?@.,acZ>{N1/G|+}EX]8UY+DmrtM%3LdB<Ohow|_G=,WkA@'],
            ['$gy6KH1T9%SX}P2@z|(Q;+P&k9U@N{*q/m]{lPc~Kw*a,oWi*5C6BFnjU',
             '+y_R=O8m8k_,l*JkuLb!g?cqM!R@!xnRpVr*>gZuU<YmPdw80!7b^{^?tuh',
             '#<XTntrXm&I8*v_7+PUOE!^t!2P}z1JHj)gXPq*wT$v^E8.^zD9Z[48y(%W5'],
            ['}i<(y2HtVNMVvG~XYN%wT~GUN&&&JO#1=h$,6w<.^&U+PKYm}+~+$l%6u%M',
             'm@yoDI|c<U#dlCE}$bFQ|t7-!$zgR)h7_zO-yn,4OruJzom3xv]^x[Aib!!',
             '#hy,8N<p/+nYg(^kV~@tQ-.FAxz!8.?ZEdE}-2C.eG9^74d_mI9t)4pj*,$e'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=DX',
             'gosntT4)N}VtGtjB]B4%*~<fu5.yVwoqX~.J3NTB]3g>3^2&CS.XF8${q|,',
             '#B2-eXks*9.Kk?RI-(??mV5*(EL^cgrGS~KFj#_.;-#AL%DrU3^v!jWV3lS{'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=DY',
             '#+#MolSLbcCH/qqnD<lsDp4hzkUx8S83/VxC2[4I)Xw]r}A,SNC4d[/2UI}6',
             'f_yyGx)k^y;B!*)jMUDT.sEFo+gpZ4ho3WZRr@OG9B5:RgU:OQsw_qd[s|a'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=DZ',
             '#f4=/5pEsv.A3~)84&ShQFPty3:|AQa;mRy1kD&fPk&_?m/TgFf8JlvIKW?X',
             '#JVgGKn(G3e.++xS>AloP638l1-WlGoUjU4P=k-Jss}5qR||N-ErPHW5(pft'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=D[',
             'OnTK|}7s%8x+PzIyp>VXS}i@JRy]D_FIi<utJzZ#)H6ZJ+%~X>ZFHK4.~_]',
             '#8oMe}lS(3k1xC<N(_xU0q%+:!>=6l_6Om+UWBJ-dc{(fwu-(Lz*tyASx~-B'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=D]',
             '#^C?ma>A9[5LLsWWn5B3DWV8.flwpVcsJGA.O-sTic=H-AC:9q1t&N80Sk>n',
             '>cG/p*R:u.kied9AJ|IH3|XkWL|CPsY@ttIy(ef(~Yymp[)R.my-0,qZ3A+'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=D^',
             '#EhY0J-T!KSQ1/!z]?p3wmSWSUdS^{o8{u%)bY*_RB=E(h!HvU2)GujiRKu6',
             '#mx-kQXjr5BGKkHvOoD-9n/q!lO54?=PCA#T_Pd/mqN+B3|$$40mfhm$1q#%'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=D_',
             '#lOq,sb)}o4io~h;}iUBa.)kX3:ghBOHH!#4}EkeYT*SqFAjSV(/hQKYa=S}',
             'wNaKZi^tIZuaek<u{pQ*]Ce6r/u5gc#n[Hv<?)dYMg2Wv&,,L|sLC_+qtwl'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=Da',
             '#;heo6SysC^V5N=l+54[R|tM-z}Bh5PAK.hpM^#Eo+g?O;N%D>5?)M=a,[00',
             ':Q<RY^>}~1Y_,gETznw32ArgGJRdmL4}50T0=;G+X1?GVNO>])&y;$q@InL'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=Db',
             '#0Ni9:o[Z1Nu$J269S#o:[uXi_PUs+9o.d7&y|T+3;f0Z%@-A^pUp/Rb=PCs',
             'gLx>/n0,GwqR:Z-1XSMK2:ba7nQ^t~NY{=/elh@,xO/KnP_W-6a&LcN8UH2'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=Dc',
             ')]0ySL)C)m$}+<3eY|XDYfSf^?czef1m5)4k;Sv6I:*bi^pxF#;QojKK5?;',
             '#ZN[-fka~}W#eQ,XU#fD}6?S&w6,h5t1*:A)3$ccJ*lkRW|DSSBvDEB^7z^q'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=Dd',
             '#<K_0SF{C:CFE)ARrlU)X<}W58eOiq/a19Nb/$WB&agqnTh80@F9Rqn,BX9h',
             '_}pZk*}a=dWzK!?n;s<~<h#Mu9mjSVuG45(y7LFPnmR/z&M.]!;3u{kR;vu'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=De',
             '#,8::KGZBoTjBW,N{r.u.aib7V{3-v+1_Ap:sq;>=s-Y6Y4ecs@6V(eac8+:',
             '#6)c8p8>,^<IS00rCM0P(v&x5ay*}M)vOVls%O2um^fVA(LPIa-#S%=>*4+.'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=Df',
             '2_E2Feuu@rA:a:pu.!A>ts<IV5$:yvSPJfpC]e,oJ,q6L5;(2JkPa]|WGkD',
             ';GRa<3:;wv3ql|^}:b+,IL6+{3A(0q26%_M~7NEJ@UMqzaQ<3?jD&%#yv|F'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=Dg',
             '>w6bCKpDx=h03EnCL>WAk8H_p?e?{7<Ziy}ShI6cfR&D(LSfr8Z.43loe<|',
             'oYa_Dm&:MbI1[T0k)hE}5SXc>eZn{M)hi{8^2]{C[662~u?Z56(z,K_B%_#'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=Dh',
             'i@7U%5OR;W)r:(mS@~b8_rLZ46fP!WK0p^9}E@ijJ-kf(mB5C];k#>eLHi_',
             '#v%Rag0g2[yno][,!T3vY]YMgM0qS{O/969Pa7N=HJK7+4(fyAS5gbWvEtiJ'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=Di',
             '/F]pWSV#h*D|kX:x@%Z~HfwA6<1}0F5@Jl*P$F|_xn29[GSwMGHAP2~,b-/',
             'ry{wTbIY3@|<sZ}]3s-?4,xeyOD*3-|RY{i,-^cYano5}lytS4!2xXwv#&4'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=Dj',
             '0S.5hvE;?q&:xy9w>/,g25JW2fGbTNSOc~tg-b}:78*Q=5OG!yr#UlaV@Mj',
             'Gw6T:~E9*X@.ErL85Qp/%=81&yvIWR?TN=mY=vhCLZP4DHMSNOnY9sz$FaW'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=Dk',
             '(3u5@Z|2?_[QsnW.5*TX>LMd/byEjh5sK;[a*BZ-zQwv3R(!ndz5KOPN|$]',
             'HhU^$p.+_#jtQWcD09A:mTZ,.C<bgH}sp4-JkIb![vUj4;a!Rx*=y69j5SA'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=Dl',
             ')/WS~1MF6%Z+c4,s}LZV8Ph<0_s@^[r{v/BNXR}Ih,@H#K-dYrrab|FL,Uv',
             'sgWg}8fQAe-$glif|K*n#z&}]+suiaNNAUMJOjJ(WtdQxZZF?Qfpx@=4iOt'],
            ['#|YCTA^%_FR2LDDKeUD<XRJu?4zDWgQ*O9tCk4h@xFgrt.~}i|mO5+h<V=Dm',
             '#?>g,R[~H^zenwA+.lO&EvJ.AU0X%mO&n4,j#y(2xT1.BeDn@9c<V0FG>mfH',
             '#U)pOQnMs${aOL~nCHxPfroK*^B9>JwVZ,ya6;fI@qW5!zNZKJCE|]=G*$^?']],
            'secp521r1/nistp521': [
            ['#',
             '#6^MdvRYdx-!l7|sVny0W(X3PJQ_Mv(|5jxgkcT<'
              +'&Bm@8M3f%VL+U3??dsA5juQ6X|g3[!c1oFT(tK5O{',
             '#ejq!OonINDDR4Y.1x[SRM*U9rHhJ+tJz!PiGwl8'
              +'8!ScmlJ@W#*/PfgVK6NIsNMNd+P&/{++vmj)Omu+e'],
            ['$',
             'H/7|Bzs(HQYe-4v(oica2kTVGN+9RS{l[JhVLzHL;'
              +'~U(hdMn8SX}NtGz_1iOR2YK#/~mKRW/vs.@=(dF',
             '#P2YwkRNpI&rasEJ<7y>V3/kUH&~u[<3#k_l%C_Jy'
              +'}n,f9reVG2!a_W?rwrt,3D:r6#NpuuxI%$}[]2Ti'],
            ['%',
             '$Xj.s|Opfj#lHCG5-!+eM57ws#+*..AP.aun1c+yp'
              +'#tS]_iVfezh7jRYeCW%ZG&cB0HaPRD!rY$JYs),.',
             '#z|nZVPU-iH9o=u?9@Es%EJ5$0O@Lg5aJc}g+AsYG'
              +'^z@kNRepbQ};kmby95>g7L$y~QYCtI,=izIOp,]B'],
            ['&',
             '@_y3gE.1(/yzrxriVoPNP:&42NaJ3H<BaAhn4)e.t'
              +'w#[>|(kvtl0k8Zn[-DIc*FC$mO49pd_Yuye0;-*',
             'lo,XTYG&u{[<ZhgUtO0+OQF}vAU&9B1#,gBH(n,2}'
              +'nA%SAw<Va3ztt1UatY>ZAYG0=V_nOGSQh)GV;2R'],
            ['(',
             'Zt776Pfc4CuZZ@u#A1J5:8@0Vd^HX3XL7MuLz;D7L'
              +'DDgH>BT4a:JPm<1gL0+.][E[oZ;K<f6TPuer>Uu',
             '$/1/}6}ek*p&:$iRx2VFus?zT!(x0Y~BOqO/.u8i>'
              +'TpK)J*Vu^cH(KF5V8TV[w1GT:vneb.1GFK,@yB9F'],
            [')',
             '%%!-xBSk)F]Qbr/r?zs_&#p-5<|%rbrweFqsBjB!w'
              +'KUOAl*le:b(wW?HxN;O!MpR{vU:U$]?e<QZsD$pN',
             '$y#gS)xEKTk0!^ePsxyC}2?k@+W{M/V2[G{da{Gxg'
              +'lW<5pp7wIoHj2krrctGANV#y]t;bq*HJ~2Y~47jr'],
            ['*',
             'R{UZ+gXq;mDpn2~zX:-dR:NNL(h0h5[FKqa#$~]Y5'
              +'pWm$@E(j+i+5Z/&yqC*ZMEJP)&v5M?9nUP1_Gqy',
             'Dl(-sUwTm~O8h]mVPV6R*T*~qJ+qJopCz@IiIGDNp'
              +'v[52zu5!_8bI0?}j2qCyCq6%#KL_8SQ==JO(]yJ'],
            ['+',
             '&OI[d{V_+$nz5A<N%ii{X1z8A3NAPtou,U[~YU6q)'
              +'w~xA*LCn8AxSx5Azqc0&!9&w[vtRG35c}X>Ok+3',
             '$33YK:L^L6BvvecSZ8z!?s?5k2NpXOakK8DE/OlDT'
              +'I|;yoB)Wo7KxX|/]>w}a}8w|;E2LG|[jf#XCrGh9'],
            [',',
             '$-3FJ;JTx@4a~k@g]kR*Of27qBU_<h%&JJX=a%}4U'
              +'&;kY.Zm*gqu|t=2Y+]OZK~lm^bB$K9:-q^*%0M/t',
             ':=nZK8qA.:C;j=Of<@Iu0@lsoO<mvYfjh:a8a[M!Q'
              +'?9F=B|W0miw|!owzlZM<)$Jr}>l1g-*~~Q<*wB['],
            ['-',
             '$LJ_V@3&iQ0b!8#75Rh*3edSjkyz:?xkL0TtV|V;c'
              +'gY}_M]6*/yO#O6102OV^am>=%/uJO-v@QCucFh=[',
             '%#Fbj:Dn|lz^[/CP[.GP#$-%$%-@n-]Nz;s/;;0In'
              +'/X1(&tE9f-zA$yrq7A2feO}qV5d7},XWQ9p7KqRh'],
            ['.',
             'qMjMmblIv+L~m7)VG?zN3lJx/rpLN(^WUECz4#ZYJ'
              +'7KTu!Xq!0&XDUfCl%);]NpbNSXjut|o#wsCsQ,+',
             'xF9/Mls*%;@egja@Y}j)<O5*R{F1#fKXAowTgLMvE'
              +'e?im>la-zS]=]FPGrl2M}ow;4s%O8x^U&%<~[~)'],
            ['/',
             '$hw=_K,lII^JSuCLz|p?*]hv84ow-s2w&|<NqKmet'
              +'ARl40CM9^nKgd=r9x?iql{=Pv0u;!Nrbv<*J6&_Y',
             '#{v:![XHfeS_>_Z0G@ebJy:?q2/LO!*<2DVU;kI$%'
              +'4E3$6GGcL~A42.ePxim*5pPww)EBZS5Q/MRrU9W3'],
            ['0',
             'jfX._&HYs5bZ@:3DdwdjLcbBg)lQ~BqieFCt:;(UO'
              +'U55#g$-x-f?Aj5m4Q.LhbR]VFks@mYdMJjRl[p#',
             '#[@a>V~,Ya75=2^DdtA72.|<?$k5dI;?.j?{d3{A]'
              +'4rbcARm^vTJ;3YB[+NiFJ|miW;$Av#bmopG5?[K<'],
            ['1',
             '$G1Byu[+EKx+wPd/n8Icspj&XM>/u?QY}ns5Ex@Y#'
              +'b_%{/a{5WZsJ(=]3]Jys}$aqu]v,*-5x/&X:,mhs',
             'V2:7w#8Q>Xb:H<+yYXdjq71B7x_,F2w-Rv?y[P82#'
              +'T/1ixjqV3j_gou}KxGq!C<yUk&r>JaoHL2tqdwM'],
            ['2',
             '_@c_9K.jB98-ey}Ig;zk:e!WB(@YJg*Iu+4^?NaGf'
              +'6fX[g=$Hk_-6Xal28(wY_!CH4H~(Yw[6Y^a,awL',
             '$b%p*WR7gR%Ss/:I(Jbr9gt6=+_+6>l475N!E/^cj'
              +'tO.X5hS5A}V/#Gq~)TAR1}*LV%;(Z{*m2p_OIGb6'],
            ['3',
             '$r,AxDQ0|!]VDf-Y]$~?(V_T&#z-P]J0=R>sZ]4xW'
              +'~c#MvGV94n^3FxY{>?VS(_T@Q,X{?P|)X(54<}T@',
             'hP?F0ot#Q9:gWwiHe#pu(j<nWKd1ypB^5g_hU)^dS'
              +'CH9PNJJh!T68e-ic1;vdcPd0d=ISn*}JkuxN!dS'],
            ['4',
             '$^]@?W_D/wy?P,z*T&?DN+8xyb#}kl@m{aG$nT(D['
              +'.!E19,#^q7g.dH%4(9#iLB+-YoW64iOP~(O>4INw',
             'nb^2)[CAVz*!(h0l/<wI?{lVoe86R|HSqp5T|O;aj'
              +'}k*T~OnzEjASj7dT&>.y)WWD:R>,KA0,OkpY)BL'],
            ['5',
             '$fB,T;=xbz05fG_XPetN{>!8#Mp~tvZ->nI@vZ9P?'
              +'=o_TUajMH1Y5t2.P7tUJ#uu~zhc^22HFa;SGjN[~',
             '#^X</O!qK*%F:bJL;Comn>dN*Pt$$,lbvJPKk#NAB'
              +'tZ@D~(S-hwj>WrPKk|KTA2}6)hvz[^Z@#xTx4FkM'],
            ['6',
             'yn/,b]~#K8B~]maDLgePR7,-Z.uqjoTd}ebGv1T1|'
              +'EO4/vb%hTt&D|g>Zv]25T2P.nV4UIh.|zASKe<4',
             '#w8;b[k:HloVi]5DZe3Dy^B<uiz#MS-W=GDWBRV~|'
              +'/-|A/[]D*DKeBoC=1ey?2sxKKAF0iT!5cO%vVIsn'],
            ['7',
             '$I^EFx_]WcXYMK7zFrRY*E{*bZgUI/BKzv]Q@PTQl'
              +'V5mm]0C]n7o^WMNt]8@-R[vcc9fGLh:<7NL~[Y(j',
             '#63!iD@7X.62g|@v_>g]?:VfFBaeHeH~3-G!C([pQ'
              +'f.OJ?m@T^^lC4lNozs2_-T-3@SpQbDjW~^I{XE3i'],
            ['=)R$U0!8,',
             '$43K{=:MACOx&!2cW>w#&lp{{V9L)y}rw+GUQu/Qe'
              +'zEj:bTg4S~ZR,^)o:Hg=>~S;SMHWqkDkn3rSl?pw',
             '#^CXXP-zV.qjuAnH$U;>e#Js&hbDMxz>>FuX9AfHx'
              +'6&q-XM;vk=q9+8XNZ1|Z|=P0kSUR<A3b-):pOqPY'],
            ['h=k])#StiU[0IMl;b,',
             '$B$wkCvI-sSc*.^7]EDil1X/IU_CX0OTD?]jA|Z^0'
              +'bBJl&two.ZcVBPXi@4]F7uYwX0VH4INDnmjfSuJ~',
             '$ompK!u&^@.^yh{XiI/U4qLw{3PWxL}2:tDLV~*Rb'
              +'yBGof?NcQJ+:S!0c/}:_r.nZo9KC9oV!~]n+FK%C'],
            ['mvX~ezP{c5jx=mCbcbal-u0~A#NX|k9DUrcah];?Q'
              +'rq8@BYmPWDf.E)~ryazD&}f/i25xsc57U]wCQT0',
             '#,Zj4Br6Cez0h8A8Du9vPFKfS{dB*fTNJ#9SB:NOX'
              +'[P^EiB^+)i+zlIwBCeh0|]0y1tA0)!W5=dD0Qt=B',
             '$m#u])*B|FgGd+J#?fjpJ~VzZ1o:Eq<7{]FxQM%.#'
              +'v|2FTnd_I>Xsc-Rp8cMU.<4@X@>>ENFmx1<X)C1-'],
            ['EzcXY<~pPs@do$o3UPnshhki>54]$5dr</6u9PD_}'
              +'cKlnH,MDoMX4O10n+CF<^J_Y$;}jFfEUCPjH,',
             '$oW|toId74g,0(b8u&q0ZJZW7g#cLx;,|&.<a#ffs'
              +'7LJ]t<E3$]b,PnZ*~|kmh;epv~YOIWo(3qWC|UXa',
             '#Wena}i+Z$gpGx6&Qh]rbGee91L{4[&7gT&Vd3i-x'
              +'bqn.GoLAv5n[uK5RJK8Sz#V2Nq%A*mD%TvYr^hst'],
            [';qyc~Ajca!_#Li9<WSQRG!+3wJ8C.1mn)E6<n#SnX'
              +'G3Up$S1<M+!ESaw+13xm@sq>8R/O:CU1U(zi;K',
             '#3Y$GMT1>u4%C&:0nTJU89P6&H_H2zwDXkzHG.)re'
              +'DD]D6*=C:8L@F[BEJX&Cf!.q=&N$ZY}T7fA;?>x_',
             '$2/n$g:-lz),3*Gcji)z#6{T|9p/2Coi$J<TUPM+!'
              +'Dnp<qcK|r,h^VD#T_fEG#x>V~i3or+f1K%%obwqH'],
            [')66_g@;43JumBGNXv^;?{=JNHkxgB%&@d6zL=I5#+'
              +'p3uvD/ceUzZg3q^Hjl8I[%E(%*[-R+Xexw@l$m',
             '#_Wl!7+c<*a(hT5xw@D_%QIq$bAMZ<}!td$6xe2y3'
              +'J<46/$,A!JGVwfHT|R4[{l=F<Rf/n{;i{qFlo}uq',
             '$gLi3Hd86fafcTuIo_G(x+i>2H:XGf?JkeI#}MRvg'
              +'n%0C@rxL?eAmzdAS2hrt8Bd#csaHUeDwu0m]5Lg#'],
            ['ET~M3_zxy:EG|WgD>ozjVD;tkxRRd7X%^tNV+n.--'
              +'v16gj^)i4%pskU/p/[#YlwFl71@/nF=wk8X',
             '#s0^]RWkPQ*(#i0k7G/Ti}+UV8ut1k<%1Vu$iMntk'
              +'Sbtua8I+:_,!zw}-CD}<-xYD6yxL>iX9;{m.Swur',
             '6[a?,#LZ3xcEUsXOByE}1%I]RQg,|ZlT@VFI>F/?0'
              +'3fzS08RysK(RnMkcfZ|9gEVtcwyHNCDa+Y|8Cy4'],
            ['+pC+{J@Pf0lQ;{~{#S;Z%D?HbiW0]CzPDvdI@g?Jl'
              +'8z[<UT3/sYL:~7Ku=6-q(,OXz}i*o0Rnx}D>q{.',
             '$VuKjXu}+jjV>c!C)ei@|i={aNAQ_KbHS>(SF}*/L'
              +'TW5;7qNppDI4p=LEvVD!e8ha%&J)fqWcP|;TEZV^',
             '#,4E*Bg6)~#[$p(R:KkJRp$I{,Bf-H#oBtA%n?}k0'
              +'4xfRAUIyjaj?=Q86@Z%(&tL|y~P=o~.>x;We{1,?'],
            ['#S6{p]=>Yj=U}$1e(O8,+Kj6?]/xZA,znh:R5rhl7'
              +'wu;VK|BD/+kx^+x_ER,ua?V/x.WHu[bP@G6~',
             '$r(tmaOoNzz+@[laHAca206Kl[=o12wT?#8TeUPK|'
              +'W9YeH8{4%A[5,4zsO+h1q:;_^aY$dY+!]AR9#gI=',
             'asjDQQ!c!xo%NCzi5X[Qx)Qi&CMK=:<^{*pzPiXG*'
              +'h%G2@;E;-_/OHl-d-zR7cLa:Rk-FH]6$3OaEn7q'],
            [';_Lqo|b(^n<.JK0Yk3bg>c+tiGpIh)%C6]H<;[4{g'
              +'F.m=eqh],ZSppR:-A;R7/#v-VO4La,Qj9kBEI:',
             '$_7{;9_9MVEHC8@u$0;V{KEC3ybH>+yc/wV_ze,8/'
              +'KA*hxF@z#b2t1]R,jj$;&-Qyt!Iozm1xIBTn5{={',
             'a.0KGy437UhJ6e=^,IFx4gnAJ@A>lL@;1.OE4r]pS'
              +'lPnwt(~_HHh@<R|8kgQ=N|pc^?u4(xgBuNu|FNk'],
            ['#Mp-#o*n2w3m>UzR8mkArtJprCbc/3=s<X19;_3hx'
              +'VW90OUS;dpg<%LTdsI4-@_vW!Jr?C@RH+M-XqcdT',
             '#Ns+$,5m&DnughQKA%2uz^FDd$98{1M5!]F)DQh*:'
              +'^F(l5$8$2s*(t{1>1y4x7>FRe/ttGAf3;u8jee!/',
             '$<%geRzL[}i(x|~{Wq~%dE^0QbeWEk{~SHsUbrL0&'
              +'XK)243VwogF4{afbi6$8K=i/bA/XVQJ*hqkGM@s*'],
            ['%%Cj07!Q*?,LNbkZIK0U|_MRmsrhfWrQ5,9v/T<RR'
              +'./F.#1:dPDW[r2V|e@-M6L2rneR#{||@W?',
             '$;mLI%es+a,ja2X^SW6#7O{g!H1>Fs6u.j(Wn;c.:'
              +'U1D&lD*zS=w%PeS+@Rf2.s)atppq)dJ(k7??N;|{',
             '#=0F6DF^,nO&8w2L]Nr>Nz/n!Ht-Y-LV}+=6.bv&m'
              +'4/Sc{b9@MZy5*GWD6}@)[]wgH=tB8qxqu{_|JB*.'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&j',
             '$I^EFx_]WcXYMK7zFrRY*E{*bZgUI/BKzv]Q@PTQl'
              +'V5mm]0C]n7o^WMNt]8@-R[vcc9fGLh:<7NL~[Y(j',
             '#xZG|H_/,uf|9hYskFf$/6urOG,)/MZ:VAqf3Y9Tl'
              +'Jh%NPtok6|^n#]pG_*6-0[QosnV%8[L.PI*,GZLX'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&k',
             'yn/,b]~#K8B~]maDLgePR7,-Z.uqjoTd}ebGv1T1|'
              +'EO4/vb%hTt&D|g>Zv]25T2P.nV4UIh.|zASKe<4',
             '#7U0&15,<7-V8*gGn~<<Pp.=}}oj*^waL&t04/>FB'
              +'$hUWc(Q{l9~KrY{zK7KL+=dUi$#G1L7PmWM1IUjS'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&l',
             '$fB,T;=xbz05fG_XPetN{>!8#Mp~tvZ->nI@vZ9P?'
              +'=o_TUajMH1Y5t2.P7tUJ#uu~zhc^22HFa;SGjN[~',
             '#O5/X=|Q9yvhg%P@2A[q]2j,m8uiS*8Uq#h;i^G&{'
              +'<:5Sp[[6-dar[Vnkn|y6x~^m.YOY=BZFN/{/mXrt'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&m',
             '$^]@?W_D/wy?P,z*T&?DN+8xyb#}kl@m{aG$nT(D['
              +'.!E19,#^q7g.dH%4(9#iLB+-YoW64iOP~(O>4INw',
             '$@,jUeC!CK!(~_4[^V2h$1Pl>x(TA_(p5YHQ}cF-]'
              +'Ethp;c_Px8aoaa+S(v*]A,&Ip+t9nTuVFVf7Fx=u'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&n',
             '$r,AxDQ0|!]VDf-Y]$~?(V_T&#z-P]J0=R>sZ]4xW'
              +'~c#MvGV94n^3FxY{>?VS(_T@Q,X{?P|)X(54<}T@',
             '$F=,A]0NbPct:/$$#~JnUHd>)2@*F94v,9Q(l,mgY'
              +'[Q-a@5eu.Zuz|f5Mwl,rUN.?&_-.E4/g(<Z/Q}yn'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&o',
             '_@c_9K.jB98-ey}Ig;zk:e!WB(@YJg*Iu+4^?NaGf'
              +'6fX[g=$Hk_-6Xal28(wY_!CH4H~(Yw[6Y^a,awL',
             'LhW^5L.{OvY.WbCE:jm4gXDX_-cAt9(R9jf1R7cS<'
              +'EG@Zy[.R]t%5$M9SG&9,4TT]@/O>&/w@7qVVW|/'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&p',
             '$G1Byu[+EKx+wPd/n8Icspj&XM>/u?QY}ns5Ex@Y#'
              +'b_%{/a{5WZsJ(=]3]Jys}$aqu]v,*-5x/&X:,mhs',
             '$V[1Os|.3eBLg>_bP+rza]9ISQq.KlrA[xBGY%E10'
              +'0@Fi(iEN>JaPMYI:1#|x:nB)]WBcZUTu*ZA30<ft'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&q',
             'jfX._&HYs5bZ@:3DdwdjLcbBg)lQ~BqieFCt:;(UO'
              +'U55#g$-x-f?Aj5m4Q.LhbR]VFks@mYdMJjRl[p#',
             '#QLhI5~:+BeydT=Gfo.J<BP>Ve~VqiiyZbxipMx&a'
              +'|!q7O1Ba|*!v$o|ZQL[Do4o9[+E6#}Sva8,rbD5)'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&r',
             '$hw=_K,lII^JSuCLz|p?*]hv84ow-s2w&|<NqKmet'
              +'ARl40CM9^nKgd=r9x?iql{=Pv0u;!Nrbv<*J6&_Y',
             '#2u1f1Ey|>GNc(@]%Dg}!T6;%W]@)3z|W*b2:tLD;'
              +'|OBuZ<h]G],}&>Yfb3Wc(A.+==$5>M$5CX}5Jh)2'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&s',
             'qMjMmblIv+L~m7)VG?zN3lJx/rpLN(^WUECz4#ZYJ'
              +'7KTu!Xq!0&XDUfCl%);]NpbNSXjut|o#wsCsQ,+',
             '$6G2X?3OZ}an;}2--+MuE4}Em6nFG2>m1-Hm}xHwG'
              +'k161$EC_havRwlxg5*XXl3m+y1SQIi>(z$Mk!C_<'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&t',
             '$LJ_V@3&iQ0b!8#75Rh*3edSjkyz:?xkL0TtV|V;c'
              +'gY}_M]6*/yO#O6102OV^am>=%/uJO-v@QCucFh=[',
             '.Ff{RYSf6~ODWW;nW(1JLCTre_Kh)Fim3EX;Ff{P$'
              +'<Drjmk*0POp3OLFEZ6%Ta^1]0d?yu].}napT0-Y'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&u',
             '$-3FJ;JTx@4a~k@g]kR*Of27qBU_<h%&JJX=a%}4U'
              +'&;kY.Zm*gqu|t=2Y+]OZK~lm^bB$K9:-q^*%0M/t',
             '$tOY-AgQCubkez^<eI/7UA/l!x<Oh<IQ|d~(=~9x>'
              +'^U<RSA3henb99IO?_0j=~,YVAE+ii;-[P)~kw*=f'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&v',
             '&OI[d{V_+$nz5A<N%ii{X1z8A3NAPtou,U[~YU6q)'
              +'w~xA*LCn8AxSx5Azqc0&!9&w[vtRG35c}X>Ok+3',
             '{Yn<RQe8mX8+!98oLQ_.Z1E,W<y{cCM>6tBG2+#if'
              +'v9}!B*h&F~8ZM3Y>$G+<xe%x}8+Q$Xyk&wd/Wv,'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&w',
             'R{UZ+gXq;mDpn2~zX:-dR:NNL(h0h5[FKqa#$~]Y5'
              +'pWm$@E(j+i+5Z/&yqC*ZMEJP)&v5M?9nUP1_Gqy',
             '$j!CYwHK05z^h|=|s4uJxFxOtvAbdh5HEPx=k8N#o'
              +'?{ve^g;.r|5Nl=!9pjSG@mkm2A{+:ic55k)VyBdw'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&x',
             '%%!-xBSk)F]Qbr/r?zs_&#p-5<|%rbrweFqsBjB!w'
              +'KUOAl*le:b(wW?HxN;O!MpR{vU:U$]?e<QZsD$pN',
             '5jb4e&}9M1~~)7;UjR=M?0mU_3o+&M*.&=#rdMLVD'
              +'=9d}qxHJm$G%^LDw(}IkYZ)UN.s)wn;Puv)misO'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&y',
             'Zt776Pfc4CuZZ@u#A1J5:8@0Vd^HX3XL7MuLz;D7L'
              +'DDgH>BT4a:JPm<1gL0+.][E[oZ;K<f6TPuer>Uu',
             '#!];hU~]wy-+gc39QRu:TZ0]AhdrGW%v9YiXGj]]!'
              +'Z%*qFYWJ6y%,j&-aDFn/C#7LyKWp8t)?,[Gg(^F{'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&z',
             '@_y3gE.1(/yzrxriVoPNP:&42NaJ3H<BaAhn4)e.t'
              +'w#[>|(kvtl0k8Zn[-DIc*FC$mO49pd_Yuye0;-*',
             '$A|?/8D{^-}Qe,4%sn|QB}{3uqJ7Pyc+gBQE.Y)=/'
              +'3(4t=B9&={:6@U1ay(kL^p$Z((ou,Po2}@J_IfMo'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&{',
             '$Xj.s|Opfj#lHCG5-!+eM57ws#+*..AP.aun1c+yp'
              +'#tS]_iVfezh7jRYeCW%ZG&cB0HaPRD!rY$JYs),.',
             '#3nY-6MmW:Ru2I&M4E)lH,%ErY<L+KoW>i:~K@!kv'
              +'Px5/B1JO4,LuI[]>Ch*%%eY)5pm4%W.Hg.*V1u#!'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&|',
             'H/7|Bzs(HQYe-4v(oica2kTVGN+9RS{l[JhVLzHL;'
              +'~U(hdMn8SX}NtGz_1iOR2YK#/~mKRW/vs.@=(dF',
             '#]Zmn!KsrXu<?qUB1MRBt>@m@Aek_Ui)faWyP>6zD'
              +'3(I4VoJiMLJOTr!Dc*Pa*mD0~Awe$,>=M%RJCo+X'],
            ['%/lHfk}Cd#z/~eykJdL_KNMWthjkU4%9hL9fS_tF>'
              +'en@bh3bsv/<8Bz-3iXbbkCg|k/xooigfaHF/Mk&}',
             '#6^MdvRYdx-!l7|sVny0W(X3PJQ_Mv(|5jxgkcT<&'
              +'Bm@8M3f%VL+U3??dsA5juQ6X|g3[!c1oFT(tK5O{',
             '#I#Vf=0T;SVjMRA_;jp-x#F$[uC%.+0mmKg|.h+1*'
              +'1@p-%9ohrT>_NchlFLzukd0?,rCG{w.mc>JV4,T]']]}

class TestEC(unittest.TestCase):
    def test_nist_testvectors(self):
        vectors = nist_test_vectors()
        for curvename in vectors:
            curve = seccure.Curve.by_name(curvename)
            for k, x, y in vectors[curvename]:
                k = seccure.deserialize_number(k, seccure.SER_COMPACT)
                x = seccure.deserialize_number(x, seccure.SER_COMPACT)
                y = seccure.deserialize_number(y, seccure.SER_COMPACT)
                self.assertEqual(curve.base * k,
                                seccure.AffinePoint(x, y, curve))

if __name__ == '__main__':
    unittest.main()
