import requests
import json
import helpers	
import sys
from requests.auth import HTTPBasicAuth
from urlparse import urljoin

API_URL = 'https://portal.senslab.info/rest/'

class Encoder(json.JSONEncoder):
    """ Encoder for serialization object python to JSON format
    """
    def default(self, obj):
        return obj.__dict__

class Api():
    """ 
    REST API
    """
    def __init__(self, url=API_URL, username=None, password=None, parser=None):
        """
        :param url: url of API. 
        :param username: username for Basic password auth
        :param password: password for Basic auth
        :param parser: command-line parser
        """
    
        self.url = url
        username, password = helpers.get_user_credentials(username, password, parser)
        self.auth = HTTPBasicAuth(username,password)
        
    def method(self, url, method='GET', data=None):
        method_url = urljoin(self.url, url)
        if (method == 'POST'):
           headers = {'content-type': 'application/json'}
           r = requests.post(method_url, data=data, headers=headers, auth=self.auth)
        elif (method == 'MULTIPART'):
           r = requests.post(method_url, files=data, auth=self.auth)
        elif (method == 'DELETE'):
           r = requests.delete(method_url, auth=self.auth) 
        else:
           r = requests.get(method_url, auth=self.auth)                    
        if (r.status_code == requests.codes.ok):
           return r.text
        # we have HTTP error (code != 200)
        else: 
           print("HTTP error code : %s \n%s" % (r.status_code, r.text)) 
           sys.exit()

    def get_sites(self):
        """ Get testbed sites description

        :returns JSONObject
        """
        return self.method('experiments?sites')

    def get_resources(self):
        """ Get testbed resources description

        :returns JSONObject
        """
        return self.method('experiments?resources')

    def get_resources_state(self):
        """ Get testbed resources state description

        :returns JSONObject
        """
        return self.method('experiments?state')


    def get_profile(self, name):
        """ Get user profile description.

        :param name: profile name
        :type name: string
        :returns JSONObject
        """
        return self.method('profile/%s' % name)

    def get_profiles(self):
        """ Get user's list profile description

        :returns JSONObject
        """
        return self.method('profiles')

    def add_profile(self, profile):
        """ Add user profile

        :param profile: profile description
        :type profile: JSONObject.
        """
        self.method('profile', method='POST', data=profile)

    def del_profile(self, name):
        """ Delete user profile

        :param profile_name: name
        :type profile_name: string
        """
        self.method('profile/%s' % name, method='DELETE')

    def submit_experiment(self, files):
        """ Submit user experiment 

        :param files: experiment description and firmware(s)
        :type files: dictionnary
        :returns JSONObject
        """
        return self.method('experiment', method='MULTIPART', data=files)

    def get_experiments(self, queryset):
        """ Get user's experiment
        :param queryset: queryset with state, limit and offset attribute
        :type queryset: string
        :returns JSONObject
        """
        return self.method('experiments?%s' % queryset)

    def get_experiments_total(self):
        """ Get the number of past, running and upcoming user's experiment.

        :returns JSONObject
        """
        return self.method('experiments?total')

    def get_experiment(self, id):
        """ Get user experiment description.

        :param id: experiment id submission (e.g. OAR scheduler) 
        :type id: string
        :returns JSONObject
        """
        return self.method('experiment/%s' % id)

    def get_experiment_resources(self, id):
        """ Get user experiment resources list description.

        :param id: experiment id submission (e.g. OAR scheduler) 
        :type id: string
        :returns JSONObject
        """
        return self.method('experiment/%s?resources' % id)

    def get_experiment_state(self, id):
        """ Get user experiment state.

        :param id: experiment id submission (e.g. OAR scheduler) 
        :type id: string
        :returns JSONObject
        """
        return self.method('experiment/%s?state' % id)

    def get_experiment_archive(self, id):
        """ Get user experiment archive (tar.gz) with description and firmware(s). 

        :param id: experiment id submission (e.g. OAR scheduler) 
        :type id: string
        :returns File
        """
        return self.method('experiment/%s?data' % id)

    def stop_experiment(self, id):
        """ Stop user experiment.

        :param id: experiment id submission (e.g. OAR scheduler) 
        :type id: string
        """
        self.method('experiment/%s' % id, method='DELETE')

    def start_command(self, id, nodes, battery=False):
        """ Launch start command on user experiment list nodes 

        :param id: experiment id submission (e.g. OAR scheduler) 
        :type id: string
        :param nodes: list of nodes
        :type nodes: JSONArray
        :param battery: powered by dc or battery 
        :type battery: boolean
        :returns JSONObject
        """ 
        if (battery):
           return self.method('experiment/%s/nodes?start&battery=true' % id, method='POST', data=nodes)
        else:
           return self.method('experiment/%s/nodes?start' % id, method='POST', data=nodes) 

    def stop_command(self, id, nodes):
        """ Launch stop command on user experiment list nodes 

        :param id: experiment id submission (e.g. OAR scheduler) 
        :type id: string
        :param nodes: list of nodes
        :type nodes: JSONArray
        :returns JSONObject
        """
        return self.method('experiment/%s/nodes?stop' % id, method='POST', data=nodes)

    def reset_command(self, id, nodes):
        """ Launch reset command on user experiment list nodes 

        :param id: experiment id submission (e.g. OAR scheduler) 
        :type id: string
        :param nodes: list of nodes
        :type nodes: JSONArray
        :returns JSONObject
        """
        print self.method('experiment/%s/nodes?reset' % id, method='POST', data=nodes)

    def update_command(self, id, files):
        """ Launch upadte command (flash firmware) on user experiment list nodes 

        :param id: experiment id submission (e.g. OAR scheduler) 
        :type id: string
        :param files: nodes list description and firmware
        :type files: dictionnary
        :returns JSONObject
        """
        print self.method('experiment/%s/nodes?update' % id, method='MULTIPART',data=files)

