'''
Argparse functions for command line utilities
'''

import argparse

def pileup2acgt(parser, subparser):
   subparser.add_argument('pileup', metavar='pileup',
                   help='pileup (SAMtools) file in input, if a gzipped file will be selected it will be opened in gzip mode, if file name is - it would be loaded from STDIN.')
   subparser.add_argument('-n', dest='n', type=int,
                   help='The minimum read depth on the base to consider the mutation on it.')
   parser_pup2muoutput      = subparser.add_argument_group(title='Output', description='Argument that involve the output destination.')
   parser_pup2muoutput.add_argument('-o', '--output', dest='output', type = str, default = '-',
                       help='Destination of the output file. To use gzip compression name the file ending by .gz. (default STDOUT).')
   parser_pup2muoutput.add_argument('--quiet', dest='quiet', action="store_true",
                       help='Do not output additional debugging.')
   parser_pup2muperformance = subparser.add_argument_group(title='Performance', description='Argument that can effect the performance.')
   parser_pup2muperformance.add_argument('-p', '--processes', dest='nproc', default="0", type=int,
                   help='Set the number of processes to split the parsing job. If it is set to 0 (default), the job will occurs with no forking to other processes. If it is bigger then 0 it is more efficient with an adequate chunk size, otherwise with smaller chuncks (eg.: < 1000) it will loose performance.')
   parser_pup2muperformance.add_argument('-c', '--chunk', dest='chunk', default="0", type=int,
                   help='Set the size (number of lines) of the portion of the file to assign to each process. If is set to 0 (default) wil set to default also the --processes parameter (-p 0). An adequate chunk size defends on the number of processes and on the file size (chunk size bigger then total number of line is not good). However a chunk size ~ 1000 leads to better performance.')
   parser_pup2muqualitysets = subparser.add_argument_group(title='Quality and Format', description='Argument that change the quality threshold or the quality format.')
   parser_pup2muqualitysets.add_argument('-q', '--qlimit', dest='qlimit', default=20,type=int,
                   help='Minimum nucleotide quality score for consider in the counts.')
   parser_pup2muqualitysets.add_argument('-f', '--qformat', dest='qformat', default="sanger",
                   help='Quality format, options are sanger or illumina, it will add an offset of 33 or 64 respectively to the qlimit value.')
   return parser.parse_args()

def pileup2abfreq(parser, subparser):
   parser_ABinput    = subparser.add_argument_group(title='Input Files',description='Required input files.')
   parser_ABgenotype    = subparser.add_argument_group(title='Genotyper',description='Options regarding the genotyping.')
   parser_ABperformance = subparser.add_argument_group(title='Performance', description='Options affecting the performance.')
   parser_ABqualitysets = subparser.add_argument_group(title='Quality and Format', description='Options that change the quality threshold and format.')
   parser_ABinput.add_argument('-r', '--reference', dest = 'reference', required = True,
                   help='The pileup of the reference/normal sample')
   parser_ABinput.add_argument('-s', '--sample', dest = 'sample', required = True,
                   help='The pileup of the tumor sample')
   parser_ABinput.add_argument('-gc', dest = 'gc', metavar = 'gc', required = True,
                   help='The GC-content file coming from UCSC genome browser, or generated in the same UCSC format')
   parser_ABqualitysets.add_argument('-q', '--qlimit', dest = 'qlimit', default = 20, type = int,
                   help='Minimum nucleotide quality score for consider in the counts. Default 20.')
   parser_ABqualitysets.add_argument('-f', '--qformat', dest = 'qformat', default = "sanger",
                   help='Quality format, options are sanger or illumina, it will add an offset of 33 or 64 respectively to the qlimit value. Default "sanger".')
   parser_ABqualitysets.add_argument('-n', dest = 'n', type = int, default = 20,
                   help='Threshold to filter positions by the sum of read depth of the two samples. Default 20.')
   parser_ABgenotype.add_argument('--hom', dest = 'hom', type = float, default = 0.9,
                   help='Threshold to select homozygous positions. Default 0.9.')
   parser_ABgenotype.add_argument('--het', dest = 'het', type = float, default = 0.25,
                   help='Threshold to select heterozygous positions. Default 0.25.')
   parser_ABperformance.add_argument('-p', '--processes', dest='nproc', default="0", type=int,
                   help='Set the number of processes to split the genotyping. If it is set to 0 (default), the job will occurs with no forking to other processes. If it is bigger then 0 it is more efficient with an adequate chunk size, otherwise with smaller chunks (eg.: < 1000) it will loose performance. Default 0')
   parser_ABperformance.add_argument('-c', '--chunk', dest='chunk', default="1", type=int,
                   help='Set the number of lines to assign to each process. If is set to 1 (default) will set to default also the --processes parameter (-p 0). An adequate chunk size defends on the number of processes and on the file size (chunk size bigger then total number of line is not good). However a chunk size ~ 1000 leads to better performance. Default 1.')
   return parser.parse_args()

def GC_windows(parser, subparser):
   subparser.add_argument('fasta', metavar = 'fasta',
                   help='the fasta file. It can be a file name or \"-\" to take the input from STDIN')
   subparser.add_argument('-w', dest = 'window', type = int, default = 50,
                   help='The window size to calculate the GC-content percentage')
   return parser.parse_args()

def merge_pileups(parser, subparser):
   subparser.add_argument('-1', '--pileup1', dest = 'p1', required = True,
                   help='The first pileup')
   subparser.add_argument('-2', '--pileup2', dest = 'p2', required = True,
                   help='The second pileup, will show as the last columns set')
   return parser.parse_args()

def reduce_abfreq(parser, subparser):
   subparser.add_argument('-a', '--abfreq', dest = 'abfreq', required = True,
                   help='An ABfreq file from the pileup2abfreq function.')
   subparser.add_argument('-w', '--window', dest = 'w', type = int, default = 50,
                   help='Window size used to binning the original ABfreq file. Default is 50.')
   return parser.parse_args()

def sequenzaExtract(parser, subparser):
   parser_io      = subparser.add_argument_group(title='Input and output',description='Input ABfreq files and output options.')
   parser_segment = subparser.add_argument_group(title='Segmentation',description='Option to control the segmentation.')
   parser_mut     = subparser.add_argument_group(title='Mutations',description='Option to filter the mutations by variant allele frequency.')
   parser_misc    = subparser.add_argument_group(title='Misc',description='Miscellaneous options.')
   parser_io.add_argument('--abfreq', dest = 'abfreq', required = True,
                   help='An existing abfreq file')
   parser_io.add_argument('-o', '--out', dest = 'dir', default = "./",
                   help='Path where to make the directory containing the processed data')
   parser_io.add_argument('-t', '--tag', dest = 'tag', required = True,
                   help='Tag to name the directory and the prefix of the generated files')
   parser_segment.add_argument('-k', '--kmin', dest = 'kmin', type = int, default = 10,
                   help='minimum number of position per segment. default 10 (WGS is suggested to set to 500 or so)')
   parser_segment.add_argument('-g', '--gamma', dest = 'gamma', type = int, default = 80,
                   help='gamma parameter for the segmentation, higher is less sensible smaller is more. default 80')
   parser_mut.add_argument('-f', '--mut-threshold', dest = 'mufreq', type = float, default = 0.1,
                   help='Threshold on the variant allele frequency to filter out the mutations. Default 0.1.')
   parser_misc.add_argument('--no-loop', dest = 'loop', action='store_false', default = True,
                   help='Boolean flag indicating if to loop over chromosomes one by one (default), or load all the file in memory')
   return parser.parse_args()

def sequenzaFit(parser, subparser):
   parser_io      = subparser.add_argument_group(title='Input',description='Input options.')
   parser_gender  = subparser.add_argument_group(title='Gender',description='Option to control the gender and X/Y chromosome.')
   parser_model   = subparser.add_argument_group(title='Model',description='Options to control the Bayesian inference.')
   parser_perf    = subparser.add_argument_group(title='Performance',description='Options to control performance.')
   parser_io.add_argument('--dir', dest = 'dir', required = True,
                   help='The directory where the data to load are stored')
   parser_io.add_argument('-t', '--tag', dest = 'tag', default = None,
                   help='Tag indicating the prefix of the data, if not specified it is assumed as the name of the specified directory')
   parser_gender.add_argument('--is-male', dest = 'isMale', action='store_true', default = False,
                   help='Boolean flag indicating if the sequencing data are from a male or female, and consequently properly handle chromosome X and Y')
   parser_perf.add_argument('-p', dest = 'ncpu', type = int, default = 4,
                   help='The number of core to use when performing the Bayesian inference. Default 4.')
   parser_gender.add_argument('-X', "--chrX", dest = 'X', type = str, default = "X",
                   help='Character defining chromosome X. Default X.')
   parser_gender.add_argument('-Y', "--chrY", dest = 'Y', type = str, default = "Y",
                   help='Character defining chromosome Y. Default Y.')
   parser_model.add_argument('-r', "--only-ratio", dest = 'onlyratio', action='store_true', default = False,
                   help='Do not take into account the BAF in the Bayesian inference, but only the depth ratio.')
   parser_model.add_argument('-f', "--segment-filter", dest = 'segfilt', type = float, default = 10e6,
                   help='Size in base-pair, to filter the segments to use in the Bayesian inference. Default 10e6.')
   parser_model.add_argument('-l', "--priors", dest = 'priors', type = str, default = '{"CN" :[1, 2, 3, 4], "value" : [1, 2, 1, 1]}',
                   help='Set the priors on the copy-number. Default 2 on CN = 2, 1 for all the other CN state \'{"CN" :[1, 2, 3, 4], "value" : [1, 2, 1, 1]}\'.')
   return parser.parse_args()

def sequenzaOverride(parser, subparser):
   parser_io      = subparser.add_argument_group(title='Input',description='Input options.')
   parser_param   = subparser.add_argument_group(title='Parameters',description='Model Parameters.')
   parser_gender  = subparser.add_argument_group(title='Gender',description='Option to control the gender and X/Y chromosome.')
   parser_io.add_argument('--dir', dest = 'dir', required = True,
                   help='The directory where the data to load are stored')
   parser_io.add_argument('-t', '--tag', dest = 'tag', default = None,
                   help='Tag indicating the prefix of the data, if not specified it is assumed as the name of the contanitor directory')
   parser_param .add_argument('-c', '--cellularity', required = True, type = float,
                   help = 'Cellularity parameter to pass to the model.')
   parser_param .add_argument('-p', '--ploidy', required = True, type = float,
                   help = 'Ploidy parameter to pass to the model.')
   parser_gender.add_argument('--is-male', dest = 'isMale', action='store_true', default = False,
                   help='Boolen flag indicating if the sequencing data are from a male or female, and consequently properly handle chromosome X and Y')
   parser_gender.add_argument('-X', "--chrX", dest = 'X', type = str, default = "X",
                   help='Character defining chromosome X. Default X.')
   parser_gender.add_argument('-Y', "--chrY", dest = 'Y', type = str, default = "Y",
                   help='Character defining chromosome Y. Default Y.')
   parser_param.add_argument('-r', "--only-ratio", dest = 'onlyratio', action='store_true', default = False,
                   help='Do not take into account the BAF in the Bayesian inference, but only the depth ratio.')
   return parser.parse_args()


class SubcommandHelpFormatter(argparse.RawDescriptionHelpFormatter):
   def _format_action(self, action):
      parts = super(argparse.RawDescriptionHelpFormatter, self)._format_action(action)
      if action.nargs == argparse.PARSER:
         parts = "\n".join(parts.split("\n")[1:])
      return parts

class DefaultHelpParser(argparse.ArgumentParser):
   def error(self, message):
      import sys
      sys.stderr.write('error: %s\n' % message)
      self.print_help()
      sys.exit(2)
