'''
Created on Dec 4, 2013

@author: scrosby
'''
import requests, json
from simple_salesforce import Salesforce

class RestClient(Salesforce):
    def __init__(self, **kwargs):
        Salesforce.__init__(self, version="29.0", **kwargs)
        
    def get_chatter_profile(self, obj='me'):
        url = self.base_url + 'chatter/users/{}'.format(obj)
        params = {}
        
        result = requests.get(url, headers=self.headers, params=params)
        
        if result.status_code != 200:
            _exception_handler(result)

        return result.json()
    

    def create_chatter_body(self, text):
        message = {"body":{
                "messageSegments":[
                    {
                        "type":"Text", 
                        "text":text}]}}
        return message

    def chatter_post(self, text):
        url = self.base_url + 'chatter/feeds/news/me/feed-items'
        body = self.create_chatter_body(text)
        
        data = json.dumps(body)
        
        result = requests.post(url, data=data, headers=self.headers)
        
        if result.status_code != 201:
            _exception_handler(result)
            
        return result.json()
    
    def chatter_on_object(self, message, obj):
        url = self.base_url + 'chatter/feeds/record/{}/feed-items'.format(obj)
        
        body = self.create_chatter_body(message)
        data = json.dumps(body)
        
        result = requests.post(url, data=data, headers=self.headers)
        
        if result.status_code != 201:
            _exception_handler(result)
            
        return result.json()
    
    def get_news_feed(self):
        url = self.base_url + 'chatter/feeds/news/me/feed-items'
        params = {}
        result = requests.get(url, headers=self.headers, params=params)
        
        if result.status_code != 200:
            _exception_handler(result)
            
        return result.json()
    
    def get_object_feed(self, obj):
        url = self.base_url + 'chatter/feeds/record/{}/feed-items'.format(obj)
        
        result = requests.get(url, headers=self.headers)
        
        if result.status_code != 200:
            _exception_handler(result)
            
        return result.json()

def _exception_handler(result, name=""):
    url = result.url
    try:
        response_content = result.json()
    except Exception:
        response_content = result.text

    if result.status_code == 300:
        message = "More than one record for {url}. Response content: {content}"
        message = message.format(url=url, content=response_content)
        raise SalesforceMoreThanOneRecord(message)
    elif result.status_code == 400:
        message = "Malformed request {url}. Response content: {content}"
        message = message.format(url=url, content=response_content)
        raise SalesforceMalformedRequest(message)
    elif result.status_code == 401:
        message = "Expired session for {url}. Response content: {content}"
        message = message.format(url=url, content=response_content)
        raise SalesforceExpiredSession(message)
    elif result.status_code == 403:
        message = "Request refused for {url}. Resonse content: {content}"
        message = message.format(url=url, content=response_content)
        raise SalesforceRefusedRequest(message)
    elif result.status_code == 404:
        message = 'Resource {name} Not Found. Response content: {content}'
        message = message.format(name=name, content=response_content)
        raise SalesforceResourceNotFound(message)
    else:
        message = 'Error Code {status}. Response content: {content}'
        message = message.format(status=result.status_code, content=response_content)
        raise SalesforceGeneralError(message)

class SalesforceMoreThanOneRecord(Exception):
    '''
    Error Code: 300
    The value returned when an external ID exists in more than one record. The
    response body contains the list of matching records.
    '''
    pass


class SalesforceMalformedRequest(Exception):
    '''
    Error Code: 400
    The request couldn't be understood, usually becaue the JSON or XML body contains an error.
    '''
    pass


class SalesforceExpiredSession(Exception):
    '''
    Error Code: 401
    The session ID or OAuth token used has expired or is invalid. The response
    body contains the message and errorCode.
    '''
    pass


class SalesforceRefusedRequest(Exception):
    '''
    Error Code: 403
    The request has been refused. Verify that the logged-in user has
    appropriate permissions.
    '''
    pass


class SalesforceResourceNotFound(Exception):
    '''
    Error Code: 404
    The requested resource couldn't be found. Check the URI for errors, and
    verify that there are no sharing issues.
    '''
    pass


class SalesforceGeneralError(Exception):
    '''
    A non-specific Salesforce error.
    '''
    pass
