'''shrinkwrap command-line utility'''
import sys
import glob
import os
import tempfile
import shutil
import subprocess
import optparse

def activate():
    '''Print the contents of scripts in ``$VIRTUAL_ENV/env.d``.'''
    VIRTUAL_ENV = os.environ['VIRTUAL_ENV']
    env_pattern = os.path.join(VIRTUAL_ENV, 'env.d', '*.sh')
    if len(glob.glob(env_pattern)) > 0:
        print '; '.join(['source ' + os.path.realpath(filename) for filename in glob.glob(env_pattern)])


def createpkg(argv):
    '''Take each listed file on the command line, copy to a temporary
    directory, rename to setup.py, run "python setup.py sdist", and copy back
    the generated tar file.
    
    Streamlines the creation of shrinkwrap packages which are entirely defined
    by a setup.py file.
    '''
    parser = optparse.OptionParser()
    parser.add_option('-p', '--package-dir', dest='package_dir', default=None,
        help='Output packages into directory hierarchy at this location')

    options, args = parser.parse_args(argv)

    orig_work_dir = os.getcwd()
    if options.package_dir is not None:
        package_dir = os.path.realpath(options.package_dir)
    else:
        package_dir = None

    for setup_filename in args:
        print 'Generating package for', setup_filename
        temp_dir = None
        try:
            temp_dir = tempfile.mkdtemp()
            target = os.path.join(temp_dir, 'setup.py')
            shutil.copy(setup_filename, target)
            os.chdir(temp_dir)
            ret = subprocess.call('python setup.py sdist', shell=True)
            if ret != 0:
                raise Exception('Error in calling setup.py for ' + setup_filename)
            file_list = glob.glob(os.path.join(temp_dir, 'dist', '*.tar.gz'))
            if len(file_list) == 0:
                raise Exception('No tar file was generated by setup.py sdist')
            elif len(file_list) > 1:
                raise Exception('Multiple files were generated by setup.py sdist')
            else:
                tar = file_list[0]
                tar_basename = os.path.basename(tar)
                if package_dir is None:
                    target = os.path.join(orig_work_dir, tar_basename)
                else:
                    last_dash = tar_basename.rfind('-')
                    package_name = tar_basename[:last_dash]
                    package_subdir = os.path.join(package_dir, package_name)
                    if not os.path.exists(package_subdir):
                        os.makedirs(package_subdir)
                    target = os.path.join(package_subdir, tar_basename)
                shutil.copy(tar, target)
        finally:
            if temp_dir is not None:
                shutil.rmtree(temp_dir)
        os.chdir(orig_work_dir)


def help():
    '''Print usage for the shrinkwrap command-line utility.'''
    print '''shrinkwrap [command]

  activate: Echo contents of scripts in $VIRTUAL_ENV/env.d to stdout.
    Should generally be used by running with backticks:
    `shrinkwrap activate`

  createpkg [filename1] ...: Take each listed file on the command line,
    copy to a temporary directory, rename to setup.py, run "python setup.py sdist",
    and copy back the generated tar file.  Streamlines the creation of shrinkwrap
    packages which are entirely defined by a setup.py file.
'''


def main(argv):
    if len(argv) < 2:
        print 'Run "%s help" for command help.' % argv[0]
        sys.exit(1)

    command = argv[1]
    if command == 'activate':
        activate()
    elif command == 'createpkg':
        createpkg(argv[2:])
    elif command == 'help':
        help()
    else:
        print 'Unrecognized options.  Run "%s help" for command help.' % argv[0]
        sys.exit(1)
