""" Captcha.Visual.Text

Text generation for visual CAPTCHAs.

SimpleCaptcha Package
Forked from PyCAPTCHA Copyright (C) 2004 Micah Dowty <micah@navi.cx>
"""
import random
from simplecaptcha import visual, file
from PIL import ImageFont, ImageDraw


class FontFactory(file.RandomFileFactory):
    """Picks random fonts and/or sizes from a given list.  'sizes' can be a
    single size or a (min,max) tuple.  If any of the given files are
    directories, all *.ttf found in that directory will be added.  """

    extensions = [".ttf"]
    base_path = "fonts"

    def __init__(self, sizes, *filenames):
        file.RandomFileFactory.__init__(self, *filenames)

        if type(sizes) is tuple:
            self.min_size = sizes[0]
            self.max_size = sizes[1]
        else:
            self.min_size = sizes
            self.max_size = sizes

    def pick(self):
        """Returns a (fileName, size) tuple that can be passed to
        ImageFont.truetype()"""
        filename = file.RandomFileFactory.pick(self)
        size = int(random.uniform(self.min_size, self.max_size) + 0.5)
        return (filename, size)

# Predefined font factories
default_font_factory = FontFactory((30, 40), "vera")


class TextLayer(visual.Layer):
    """Represents a piece of text rendered within the image.  Alignment is
    given such that (0,0) places the text in the top-left corner and (1,1)
    places it in the bottom-left.

    The font and alignment are optional, if not specified one is chosen
    randomly. If no font factory is specified, the default is used.  """

    def __init__(self, text,
                 alignment    = None,
                 font         = None,
                 font_factory = None,
                 text_color   = "black",
                 border_size  = 0,
                 border_color = "white",
                 ):
        if font_factory is None:
            global default_font_factory
            font_factory = default_font_factory

        if font is None:
            font = font_factory.pick()

        if alignment is None:
            alignment = (random.uniform(0, 1),
                         random.uniform(0, 1))

        self.text        = text
        self.alignment   = alignment
        self.font        = font
        self.text_color   = text_color
        self.border_size  = border_size
        self.border_color = border_color

    def render(self, img):
        font = ImageFont.truetype(*self.font)
        text_size = font.getsize(self.text)
        draw = ImageDraw.Draw(img)

        # Find the text's origin given our alignment and current image size
        x = int((img.size[0] - text_size[0] - self.border_size*2)
                * self.alignment[0] + 0.5)
        y = int((img.size[1] - text_size[1] - self.border_size*2)
                * self.alignment[1] + 0.5)

        # Draw the border if we need one. This is slow and ugly, but there
        # doesn't seem to be a better way with PIL.
        if self.border_size > 0:
            for bx in (-1, 0, 1):
                for by in (-1, 0, 1):
                    if bx and by:
                        draw.text((x + bx * self.border_size,
                                   y + by * self.border_size),
                                  self.text, font=font, fill=self.border_color)

        # And the text itself...
        draw.text((x, y), self.text, font=font, fill=self.text_color)
