# -*- Mode: python; tab-width: 4; indent-tabs-mode: nil; coding: utf-8 -*-
"""
simpledropbox.py
~~~~~~~~~~~~~~~~

..
    :copyright: 2010 Serge Émond
    :license: Apache License 2.0

"""

__all__ = [
    'SdFile', 'SdDirectory', 'SdObject',
    'SimpleDropbox',
    'SdbError', 'UnexpectedResultError',
]

import re
import os.path

import logging

from urllib import urlencode, quote
import urllib2
import cookielib
# To build multipart post data in put()
from poster.encode import multipart_encode, MultipartParam
from poster.streaminghttp import register_openers

# Time-related
import datetime
import time
import pytz
# import timelib
import random
from email.utils import parsedate

# To guess mime type on put()
import mimetypes

# To normalize put filenames to NFC
import unicodedata

try:
    from cStringIO import StringIO
except ImportError:
    from StringIO import StringIO

from BeautifulSoup import BeautifulSoup

from .types import SdObject, SdFile, SdDirectory

register_openers()

class SdbError(Exception):
    def __init__(self, m):
        self.message = m

class UnexpectedResultError(Exception):
    def __init__(self, m):
        self.message = m


class SimpleDropbox(object):
    
    urlbase = 'https://www.dropbox.com/'
    dlbase = 'https://dl-web.dropbox.com/'
    
    default_headers = {
        # 'User-Agent': 'Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; en-US; rv:1.9.2) Gecko/20100115 Firefox/3.6',
    }
    
    _dtformat = '%m/%d/%y %I:%M%p'
    _tzname = time.tzname[0]
    
    _re_dateclean = re.compile(ur'[^\x21-\x7f]+', re.U)
    
    db_uid = None
    db_token = None
    
    # Result of the last command
    last_result = None
    
    def __init__(self, login, password, tzname=None, dtformat=None):
        """docstring for __init__"""
        self._login = unicode(login)
        self._password = unicode(password)
        
        if tzname is not None:
            self._tzname = tzname
        if dtformat is not None:
            self._dtformat = str(dtformat)
        
        self._tz = pytz.timezone(self._tzname)
        
        self.log = logging.getLogger('%s.%s' % (__name__, self.__class__.__name__))
        
        self.cookiejar = cookielib.LWPCookieJar()
        
        debug = False
        handlers = [
            urllib2.HTTPHandler(debuglevel=debug),
            urllib2.HTTPSHandler(debuglevel=debug),
            urllib2.HTTPCookieProcessor(self.cookiejar),
        ]
        
        opener = urllib2.build_opener(*handlers)
        urllib2.install_opener(opener)
    
    _re_constants_block = re.compile(ur'<script.*?>.{0,4}var Constants = \{(?P<constants>.*?)\};', re.S|re.U)
    def _extract_constants(self, html):
        m = self._re_constants_block.search(html)
        if not m:
            raise Exception("Error extracting constants")
        
        c_block = m.group('constants').strip()
        items = c_block.split(u',')
        for item in items:
            item = item.strip()
            pieces = item.split(u':')
            key = pieces.pop(0).strip()
            value = u':'.join(pieces).strip(" '\"")
            if key == 'uid':
                self.db_uid = value
            elif key == 'TOKEN':
                self.db_token = value
        
        return
    
    
    def gen_job_id(self):
        """Generate a job ID similar to dropbox's ajax libs."""
        now = datetime.datetime.utcnow()
        
        return '%d%d%06d' % (
            time.mktime(now.timetuple()),
            now.microsecond * 1e-3,
            random.randint(0, 999999),
        )
    
    
    _re_remove_numbering = re.compile(ur'(?P<n>.*) \((?P<v>[0-9]+)\)?')
    def remove_numbering(self, name):
        """
        Remove dropbox's auto-numbering from the name.
        
        Returns a tuple: (base_name, numbering)
        
        Ex:
         * remove_numbering('abc (13)') returns ('abc','13')
         * remove_numbering('abc') returns ('abc', None)
        
        """
        m = self._re_remove_numbering.match(name)
        if not m:
            return (name, None)
        return (m.group('n'), m.group('v'))
    
    
    def cmd(self, command, src=None, arguments=None, with_job_id=False):
        """
        Run a command.
        
        This works for *most* commands.
        
        arguments can be a dict, or a list of tuples.
        
        src is a special argument. If given, is sent as a GET argument.
        It is treated as a filename and passed through _file_path()
        
        """
        
        if src is None:
            src = ''
        else:
            if not isinstance(src, str):
                src = unicode(src).encode('utf-8')
        
        url = self.urlbase + 'cmd/%(command)s%(src)s?long_running' % {
            'command': command,
            'src': src,
        }
        
        if isinstance(arguments, dict):
            arguments = zip(arguments.iterkeys(), arguments.itervalues())
        
        post_data = []
        
        if arguments is not None:
            for arg in arguments:
                arg_k = arg[0]
                arg_v = arg[1]
                if isinstance(arg_k, unicode):
                    arg_k = arg_k.encode('utf-8')
                if isinstance(arg_v, unicode):
                    arg_v = arg_v.encode('utf-8')
                post_data.append((arg_k, arg_v))
        
        post_data.append(('t', self.db_token))
        if with_job_id:
            post_data.append(('job_id', self.gen_job_id()))
        post_data.append(('_', ''))
        
        post_data = urlencode(post_data, doseq=1)
        
        headers = self.default_headers.copy()
        req = urllib2.Request(url=url, data=post_data, headers=headers)
        self.cookiejar.add_cookie_header(req)
        page = urllib2.urlopen(req)
        pdata = page.read()
        
        self.last_result = pdata
        
        self.log.debug(u"command '%s' result: %s" % (command, pdata))
        
        return pdata.strip()
    
    
    def login(self):
        """Log in the website."""
        self.log.debug("Fetching login page")
        
        headers = self.default_headers.copy()
        req = urllib2.Request(url=self.urlbase, headers=headers)
        self.cookiejar.add_cookie_header(req)
        page = urllib2.urlopen(req)
        pdata = page.read()
        
        soup = BeautifulSoup(pdata)
        
        formvals = []
        formvals.append(('t', soup.find(attrs={'name':'t', 'type':'hidden'}).get('value')))
        formvals.append(('login_email', self._login))
        formvals.append(('login_password', self._password))
        data = urlencode(formvals)
        
        self.log.info("Logging in")
        
        headers = self.default_headers.copy()
        headers['Referer'] = self.urlbase
        req = urllib2.Request(url=self.urlbase+'login', data=data, headers=headers)
        self.cookiejar.add_cookie_header(req)
        page = urllib2.urlopen(req)
        pdata = page.read()
        
        self.default_headers['Referer'] = self.urlbase + 'home'
        
        self._extract_constants(pdata)
        
        if not self.db_uid or not self.db_token:
            raise Exception("Error logging in")
    
    
    def ls(self, path, filter=None):
        """List the contents of a directory.
        
        :param path: An object that can be converted to unicode.
        :param filter: SdDirectory, SdFile, or to get both, (SdObject or None)
        
        .. note::
        
            All paths must be "absolute" (e.g. begin with '/')
        
        This method returns a list of SdObject.
        
        To convert to a pure list of file names:
        
            names = [unicode(f) for p in paths]
        
        If the directory is empty or if it doesn't exist,
        returns an empty list.
        
        """
        path = unicode(path)
        
        if filter not in (SdFile, SdDirectory):
            filter = SdObject
        
        # Maybe I should throw an exception instead..
        if path[0] != u'/':
            path = u'/' + path
        
        url = self.urlbase + 'browse2' + quote(path.encode('utf-8'))
        
        get_data = urlencode({'ajax':'yes'})
        url = url + '?' + get_data
        
        post_data = urlencode((
            ('d', 1),
            ('mini', ''),
            ('t', self.db_token),
            ('_', ''),
        ))
        
        self.log.info("ls %r" % path)
        
        headers = self.default_headers.copy()
        req = urllib2.Request(url=url, data=post_data, headers=headers)
        self.cookiejar.add_cookie_header(req)
        page = urllib2.urlopen(req)
        pdata = page.read()
        
        if pdata.find('Folder is empty.') != -1:
            return []
        
        soup = BeautifulSoup(pdata, convertEntities=BeautifulSoup.HTML_ENTITIES)
        s_files = soup.findAll(attrs={'class':'browse-file-box-details'})
        f_files = []
        
        for s_file in s_files:
            file_el = s_file.find(attrs={'class':'details-filename'}).a
            browse_path = dict(file_el.attrs)['href']
            
            if browse_path[0:14] == u'/browse_plain/':
                f_file = SdDirectory()
            else:
                f_file = SdFile()
            
            if not isinstance(f_file, filter):
                continue
            
            f_file['path'] = os.path.join(path, unicode(file_el.string))
            
            if isinstance(f_file, SdFile):
                f_file['get_url'] = browse_path
                
                # size_string = s_file.find(attrs={'class':'details-size'}).string,
                # f_file['size'] = str(self._re_dateclean.sub(' ', unicode(size_string)))
                # 
                # dt_string = unicode(s_file.find(attrs={'class':'details-modified'}).string)
                # dt_string = str(self._re_dateclean.sub(' ', dt_string))
                # 
                # try:
                #     dt = datetime.datetime.strptime(dt_string, self._dtformat)
                # except ValueError:
                #     dt = datetime.datetime(*time.gmtime(timelib.strtotime(dt_string))[:6])
                # self._tz.localize(dt)
                # f_file['modified'] = self._tz.localize(dt)
            
            f_files.append(f_file)
        
        return f_files
    
    
    def exists(self, path):
        """Similar to ls, but return the info about a single file/dir.
        
        Returns SdFile/SdDirectory, or None.
        
        """
        dirname, name = os.path.split(unicode(path))
        
        self.log.info("exists %r" % path)
        
        dlist = self.ls(dirname)
        
        if len(dlist) < 1:
            return None
        
        for entry in dlist:
            if entry['path'] == path:
                return entry
        
        return None
    
    def ls_file(self, path):
        """Similar to ls, but return the info about a single *file* (not dir).
        
        :param path: A path, can be a "unicodable" obj or SdFile.
        
        It doesn't make sence to pass a SdDirectory object.
        
        Returns either a SdFile, or None.
        
        """
        if isinstance(path, SdDirectory):
            return None
        
        if isinstance(path, SdFile):
            return path
        
        dirname, name = os.path.split(unicode(path))
        
        self.log.info("stat %r" % path)
        
        dlist = self.ls(dirname, filter=SdFile)
        
        if len(dlist) < 1:
            return None
        
        for entry in dlist:
            if entry['path'] == path:
                return entry
        
        return None
    
    def get(self, path):
        """Get a single file.
        
        Returns a tuple: ``(info, data)``.
        
        ``info`` is a SdFile object, data the contents.
        
        If the file was not found, returns None
        
        """
        if not isinstance(path, SdFile):
            # We must find the url first.. annoying :|
            path = self.ls_file(path)
        
        if not isinstance(path, SdFile):
            # Not found..
            return None
        
        url = path['get_url']
        
        self.log.info(u"get %s" % unicode(path))
        
        headers = self.default_headers.copy()
        req = urllib2.Request(url=url, headers=headers)
        self.cookiejar.add_cookie_header(req)
        page = urllib2.urlopen(req)
        pdata = page.read()
        
        info = page.info()
        
        f = SdFile()
        f.modified = pytz.utc.localize(datetime.datetime(*parsedate(info['date'])[:6]))
        f.size = int(info['content-length'])
        f.etag = info['etag']
        
        return f, pdata
    
    def stat(self, path):
        """Get a single file's info.
        
        Returns a SdFile object.
        
        If the file wasn't found, returns None.
        
        """
        if not isinstance(path, SdFile):
            # We must find the url first.. annoying :|
            path = self.ls_file(path)
        
        if not isinstance(path, SdFile):
            # Not found..
            return None
        
        url = path['get_url']
        
        self.log.info(u"head %s" % unicode(path))
        
        class Req(urllib2.Request):
            def get_method(self):
                return 'HEAD'
        
        headers = self.default_headers.copy()
        req = Req(url=url, headers=headers)
        page = urllib2.urlopen(req)
        
        info = page.info()
        
        path.modified = pytz.utc.localize(datetime.datetime(*parsedate(info['date'])[:6]))
        path.size = int(info['content-length'])
        path.etag = info['etag']
        
        return path
    
    def put(self, path, data):
        """
        Upload a file.
        
        :param path: Destination path
        :param data: Data, can be the data itself or a generator (e.g. open())
        
        Like most other dropbox calls, this creates inexistent subdirectories.
        
        """
        path = unicodedata.normalize('NFC', unicode(path))
        dst, name = os.path.split(path)
        
        headers = self.default_headers.copy()
        
        if isinstance(data, unicode):
            data = data.encode('utf-8')
        if isinstance(data, str):
            data = StringIO(data)
        
        post_data = []
        post_data.append(('t', self.db_token))
        post_data.append(('plain', 'yes'))
        post_data.append(('dest', dst.encode('utf-8')))
        mp = MultipartParam(
            'file',
            # filename=str(name.encode('utf-8')),
            fileobj=data,
            filetype=mimetypes.guess_type(name)[0] or 'application/octet-stream',
        )
        # Hack: poster forces either ascii,xmlcharrefreplace or string_escape
        # which is totally useless
        mp.filename = name.encode('utf-8')
        mp.filename = mp.filename.replace('"', '\\"')
        post_data.append(mp)
        
        
        datagen, dataheads = multipart_encode(post_data)
        headers.update(dataheads)
        
        self.log.info(u"put %s", path)
        
        # poster's example pass datagen directly to urllib2
        # however it seems it doesn't support len()
        data = ''.join(datagen)
        
        req = urllib2.Request(url=self.dlbase + 'upload', data=data, headers=headers)
        self.cookiejar.add_cookie_header(req)
        page = urllib2.urlopen(req)
        pdata = page.read()
        
        return True
    
    
    # ################ cmd() based commands
    
    def cp(self, files, to):
        """Copy one or more files to a single destination.
        
        :param files: a list of items convertible to unicode
        :param to: an object convertible to unicode, hopefully a directory
        
        .. note::
        
            All paths must be "absolute" (e.g. begin with '/')
        
        """
        if not isinstance(files, (list, tuple)):
            files = [files]
        
        files = [unicode(f) for f in files]
        args = [('files', f) for f in files]
        args.append(('to_path', unicode(to)))
        
        self.log.debug("cp %r %r" % (files, to))
        
        res = self.cmd('copy', arguments=args, with_job_id=True)
        
        if res[0:4] == 'err:':
            raise SdbError(res[4:])
        
        if res != 'ok':
            raise UnexpectedResultError(u"Expected 'ok', got: %s" % (res,))
    
    def mv(self, files, to):
        """Move one or more files to a single destination.
        
        :param files: a list of items convertible to unicode
        :param to: an object convertible to unicode, hopefully a directory
        
        .. note::
        
            All paths must be "absolute" (e.g. begin with '/')
        
        """
        if not isinstance(files, (list, tuple)):
            files = [files]
        
        files = [unicode(f) for f in files]
        args = [('files', f) for f in files]
        args.append(('to_path', unicode(to)))
        
        self.log.debug("mv %r %r" % (files, to))
        
        res = self.cmd('move', arguments=args, with_job_id=True)
        
        if res[0:4] == 'err:':
            raise SdbError(res[4:])
        
        if res != 'ok':
            raise UnexpectedResultError(u"Expected 'ok', got: %s" % (res,))
        
    
    def rm(self, files):
        """Delete one or more files.
        
        :param files: a list of items convertible to unicode
        
        .. note::
        
            All paths must be "absolute" (e.g. begin with '/')
        
        """
        if not isinstance(files, (list, tuple)):
            files = [files]
        
        files = [unicode(f) for f in files]
        args = [('files', f) for f in files]
        
        self.log.debug("rm %r" % (files,))
        
        res = self.cmd('delete', arguments=args, with_job_id=True)
        
        if res[0:4] == 'err:':
            raise SdbError(res[4:])
        
        if res != 'ok':
            raise UnexpectedResultError(u"Expected 'ok', got: %s" % (res,))
        
    
    def rename(self, path, to):
        """
        Rename a single file.
        
        :param path: an object convertible to unicode
        :param to: an object convertible to unicode
        
        .. note::
        
            All paths must be "absolute" (e.g. begin with '/')
        
        """
        path = unicode(path)
        
        args = []
        args.append(('to_path', unicode(to)))
        # Ajax: this is set to "yes" when path is a folder?
        args.append(('folder', ''))
        # What the heck is this?
        args.append(('editorId', ''))
        
        self.log.debug("rename %r %r", path, to)
        
        res = self.cmd('rename', src=path, arguments=args)
        
        if res[0:4] == 'err:':
            raise SdbError(res[4:])
        
        res_path, node = res.split(':')
        eres_path = os.path.join(os.path.dirname(path), to)
        
        # If the destination exists, they add ' (##)'
        
        res_base, res_num = self.remove_numbering(res_path)
        eres_base, eres_num = self.remove_numbering(eres_path)
        
        if res_base != eres_base:
            self.log.error(u"Expected '%s:NODE', got %s" % (eres_path, res))
            raise UnexpectedResultError(u"Expected '%s:NODE', got: %s" % (eres_path, res))
        
        return node
    
    def mkdir(self, path):
        """Create a directory.
        
        :param path: an object convertible to unicode
        
        Will create missing directories recursively.
        
        .. note::
        
            All paths must be "absolute" (e.g. begin with '/')
        
        """
        path = unicode(path)
        
        mkdir_in, name = os.path.split(path)
        
        args = []
        args.append(('to_path', unicode(name)))
        args.append(('folder', 'yes'))
        # What the heck is this?
        args.append(('editorId', ''))
        
        self.log.debug("mkdir %r", path)
        
        res = self.cmd('new', src=mkdir_in, arguments=args)
        
        if res[0:4] == 'err:':
            raise SdbError(res[4:])
        
        res_path, node = res.split(':')
        # Seems to be a bug with dropbox right now:
        # if mkdir "/a/b/c", returns "/a/b/a/b/c"
        eres_path = os.path.join(mkdir_in, path[1:])
        if res_path != eres_path:
            self.log.error(u"Expected '%s:NODE', got %s" % (eres_path, res))
            raise UnexpectedResultError(u"Expected '%s:NODE', got: %s" % (eres_path, res))
        
        return node
    

