#!/usr/bin/env python

from __future__ import absolute_import, division
import math

class Vector(object):
    '''This is a simple class representing a 3d vector. This class is 
    mutable -- you can manipulate the x, y, and z components.'''
    def __init__(self, x, y, z):
        self.x = x
        self.y = y
        self.z = z
        self._cached_magnitude = math.sqrt(self.x**2 + self.y**2 + self.z**2)
        
    @property
    def magnitude(self):
        '''Returns the magnitude.'''
        return self._cached_magnitude
    
    @magnitude.setter
    def magnitude(self, value):
        '''Sets the new magnitude relative to the origin, and adjusts 
        x, y, and z accordingly.'''
        ratio = value / self._cached_magnitude
        self.x *= ratio
        self.y *= ratio
        self.z *= ratio
        self._cached_magnitude = value
        
    def dot(self, other):
        '''Computes the dot product of this vector against another, and 
        returns a new vector.'''
        return Vector(self.x * other.x, self.y * other.y, self.z * other.z)
        
    def cross(self, other):
        '''Computes the cross product of this vector against another, and 
        returns a new vector.'''
        f = lambda a, b, c, d: a*d - b*c
        x = f(self.y, self.z, 
              other.y, other.z)
        y = f(self.x, self.z,
              other.x, other.z)
        z = f(self.x, self.y,
              other.x, other.y)
        return Vector(x, -y, z)
        
    def __repr__(self):
        return 'Vector({0}, {1}, {2})'.format(self.x, self.y, self.z)
        
    def __str__(self):
        return '({0}, {1}, {2})'.format(self.x, self.y, self.z)
        
    def __eq__(self, other):
        if not isinstance(other, Vector):
            return False
        else:
            return (self.x == other.x) and (self.y == other.y) and (self.z == other.z)
            
    def __neq__(self, other):
        return not self.__eq__(other)
        
    def __hash__(self):
        return hash((self.x, self.y, self.z))
        
    def __len__(self):
        return 3
        
    def tuple(self):
        return (self.x, self.y, self.z)
        
    def list(self):
        return [self.x, self.y, self.z]
        
    def __getitem__(self, key):
        if key == 0:
            return self.x
        elif key == 1:
            return self.y
        elif key == 2:
            return self.z
        else:
            raise IndexError("Vector objects contain only 3 elements. Index out of range.")
            
    def __setitem__(self, key, value):
        if key == 0:
            self.x = value
        elif key == 1:
            self.y = value
        elif key == 2:
            self.z = value
        else:
            raise IndexError("Vector objects contain only 3 elements. Index out of range.")
            
    def __iter__(self):
        return self.tuple().__iter__()
        
    def __reversed__(self):
        return (self.z, self.y, self.x).__iter__
        
    def __add__(self, other):
        if isinstance(other, (int, float)):
            return Vector(self.x + other, self.y + other, self.z + other)
        elif isinstance(other, Vector):
            return Vector(self.x + other.x, self.y + other.y, self.z + other.z)
        else:
            raise TypeError("Cannot add Vector and {0}".format(type(other)))
            
    def __radd__(self, other):
        return self.__add__(other)
            
    def __sub__(self, other):
        if isinstance(other, (int, float)):
            return Vector(self.x - other, self.y - other, self.z - other)
        elif isinstance(other, Vector):
            return Vector(self.x - other.x, self.y - other.y, self.z - other.z)
        else:
            raise TypeError("Cannot subtract Vector and {0}".format(type(other)))
    
    def __rsub__(self, other):
        return self.__rsub__(other)
        
    def __mul__(self, other):
        if isinstance(other, (int, float)):
            return Vector(self.x * other, self.y * other, self.z * other)
        elif isinstance(other, Vector):
            return self.dot(other)
        else:
            raise TypeError("Cannot multiply Vector and {0}".format(type(other)))
            
    def __rmul__(self, other):
        return self.__mul__(other)
            
    def __div__(self, other):
        if isinstance(other, (int, float)):
            return Vector(self.x / other, self.y / other, self.z / other)
        else:
            raise TypeError("Cannot divide Vector and {0}".format(type(other)))
            
    def __truediv__(self, other):
        return self.__div__(other)
        
    def __neg__(self):
        return Vector(-self.x, -self.y, -self.z)
        
    def __abs__(self):
        return self._cached_magnitude