// Copyright (c) 2012. Los Alamos National Security, LLC. 

// This material was produced under U.S. Government contract DE-AC52-06NA25396
// for Los Alamos National Laboratory (LANL), which is operated by Los Alamos 
// National Security, LLC for the U.S. Department of Energy. The U.S. Government 
// has rights to use, reproduce, and distribute this software.  

// NEITHER THE GOVERNMENT NOR LOS ALAMOS NATIONAL SECURITY, LLC MAKES ANY WARRANTY, 
// EXPRESS OR IMPLIED, OR ASSUMES ANY LIABILITY FOR THE USE OF THIS SOFTWARE.  
// If software is modified to produce derivative works, such modified software should
// be clearly marked, so as not to confuse it with the version available from LANL.

// Additionally, this library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License v 2.1 as published by the 
// Free Software Foundation. Accordingly, this library is distributed in the hope that 
// it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See LICENSE.txt for more details.

//--------------------------------------------------------------------------
// File:    FileSizeLimiter.h
// Module:  File
// Author:  Keith Bisset
// Created: July 29 2003
//
// Description: Limit the size of output files.  When the specified
// file size is exceeded, close the stream and reopen with a new name.
// The method checkSize() should be called whenever a file can be
// split, typically after each record.
//
// @@
//
//--------------------------------------------------------------------------

#ifndef NISAC_FILE_FILESIZELIMITER
#define NISAC_FILE_FILESIZELIMITER
#include <iostream>
#include <string>
#include <sstream>
#include "simx/File/fstream.h"

namespace File {
/// \brief The rename policy specifes the name to use to open subsequent
/// output files.
///
/// The policy class should contain a function with the
/// following name and signature:
///
/// static std::string newName(const std::string& base, int numFiles);
///
///  base is the base filename, and numFiles is the number of the
///  current file, starting at 0.  It should return a unique name for
///  each number.  
namespace Policy
{

/// \class Base26 FileSizeLimiter.h "File/FileSizeLimiter.h"
///
/// \brief The Base26 policy uses base26 for output file extensions (.AA, .AB, etc).
class Base26
{
public:
  static std::string newName(const std::string& base, int numFiles);
};

inline std::string Policy::Base26::newName(const std::string& base, int numFiles)
{
  std::stringstream newname;
  newname << base << '.'
          << (char)(numFiles/26+'A')
          << (char)(numFiles%26+'A')
          << std::ends;
  return newname.str();
}

/// \class Number FileSizeLimiter.h "File/FlieSizeLimiter.h"
///
/// \brief The Number policy numbers the output files.
///
/// Width specifies the number of digits to use.
template <int Width=2>
class Number
{
public:
  static std::string newName(const std::string& base, int numFiles);
};

template<int Width>
inline std::string Policy::Number<Width>::newName(const std::string& base, int numFiles)
{
  std::stringstream newname;
  newname << base << '.';
  newname.width(Width);
  newname.fill('0');
  newname << numFiles
          << std::ends;
  return newname.str();
}

} // namespace Policy

template < class RenamePolicy = Policy::Base26 >
class FileSizeLimiter;
typedef FileSizeLimiter<> DefaultFileSizeLimiter;

/// \class FileSizeLimiter FileSizeLimiter.h "File/FileSizeLimiter.h"
///
/// \brief Limit the size of an output file by transparently closing a
/// stream and reopening it with a different file.
template < class RenamePolicy >
class FileSizeLimiter
{
public:
  /// Construct a limiter.
  /// \param os fstream to monitor, automatically opened
  /// \param basefilename Base filename to write to
  /// \param limit close the file when over this many bytes
  ///
  /// The file stream is opened with the unadorned filename.  If the
  /// file size limit is exceeded, the first file is renamed with the
  /// first extension added to the basename, and a new file opened with
  /// the second extension added to the basename.
  FileSizeLimiter(std::ofstream& os, const std::string& basefilename, int limit, std::string header="# ");
  ~FileSizeLimiter();

  /// Checks the current file size against the limit.  If the file is
  /// greater than the limit, the file is closed and a file with a new
  /// name is opened.  The new name is generated by the specified
  /// policy class.  True is returned if a new file was opened.
  bool checkSize();

  // Return the name of the currently opened file.
  std::string currentFilename();

  // set the header string.
  void setHeader(std::string hdr);

  // Return the header string
  std::string getHeader();
  
protected:
private:
  std::string newName(const std::string& base, int numFiles);
  
  std::ofstream &fFile;
  const std::string fBaseFilename; 
  const int fLimit;
  int fNumFiles;
  std::string fHeader;
};

template < class RenamePolicy >
inline FileSizeLimiter<RenamePolicy>::FileSizeLimiter(std::ofstream& file,
                                                      const std::string& basefilename,
                                                      int limit, std::string header)
  : fFile(file),
    fBaseFilename(basefilename),
    fLimit(limit),
    fNumFiles(0),
	fHeader(header)
{
  fFile.open(fBaseFilename.c_str());
}

template < class RenamePolicy >
inline FileSizeLimiter<RenamePolicy>::~FileSizeLimiter()
{
}

template < class RenamePolicy >
inline std::string FileSizeLimiter<RenamePolicy>::currentFilename()
{
  return RenamePolicy::newName(fBaseFilename, fNumFiles);
}

template < class RenamePolicy >
inline void FileSizeLimiter<RenamePolicy>::setHeader(std::string hdr)
{
  fHeader = hdr;
}

template < class RenamePolicy >
inline std::string FileSizeLimiter<RenamePolicy>::getHeader()
{
  return fHeader;
}

template < class RenamePolicy >
inline bool FileSizeLimiter<RenamePolicy>::checkSize()
{
  if (fFile.tellp() < fLimit) {
    return false;
  }

  if (fNumFiles == 0)           // no extension
  {
    std::string newname = RenamePolicy::newName(fBaseFilename, fNumFiles);
    std::rename(fBaseFilename.c_str(), newname.c_str());
  }
  fNumFiles++;
  std::string newname = RenamePolicy::newName(fBaseFilename, fNumFiles);
  fFile.close();
  fFile.open(newname.c_str());
  fFile << fHeader << std::endl;
  return true;
}

} // namespace
#endif // NISAC_FILE_FILESIZELIMITER
