"""siteadm - per user apache config snippet management tool

Copyright (c) 2009, Florian Wagner <florian@wagner-flo.net>.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

"""

import os
import os.path as path
import tempfile
import shutil
import subprocess
import argparse
import sys

from siteadm import apacheconf
from siteadm import transform

def backup_config(vhostdir, prefix):
    tempdir = tempfile.mkdtemp()

    for item in os.listdir(vhostdir):
        if not item.startswith(prefix):
            continue
        
        shutil.move(path.join(vhostdir, item), tempdir)
        
    return tempdir

def revert_config(vhostdir, prefix, tempdir):
    for item in os.listdir(vhostdir):
        if not item.startswith(prefix):
            continue

        os.unlink(path.join(vhostdir, item))

    for item in os.listdir(tempdir):
        shutil.move(path.join(tempdir, item), vhostdir)

    os.rmdir(tempdir)

def remove_backup(tempdir):
    for item in os.listdir(tempdir):
        os.unlink(path.join(tempdir, item))

    os.rmdir(tempdir)

def cmd_cdiff(ns):
    config = apacheconf.Config(ns.file)
    transform.transform_config(config)

    pipe = subprocess.Popen(
        'diff -u "%s" - | colordiff' % ns.file,
        shell=True, stdin=subprocess.PIPE, stdout=subprocess.PIPE)
    pipe.stdin.write(config.dump())
    pipe.stdin.close()
    sys.stdout.write(pipe.stdout.read())

    return 0

def cmd_reload(ns):
    prefix = os.getenv('USER') + '-'
    sitesdir = path.join(os.getenv('HOME'), 'sites')
    vhostdir = path.join('/etc/apache2/local.d/vhosts.d')
    configs = []

    # parse and transform new config
    for item in os.listdir(sitesdir):
        if not item.endswith('.vhost'):
            continue
        
        config = apacheconf.Config(path.join(sitesdir, item))
        transform.transform_config(config)

        configs.append((config, item))

    # move users config out of the way
    tempdir = backup_config(vhostdir, prefix)
            
    # generate config
    for config,filename in configs:
        fp = file(path.join(vhostdir, prefix + filename), 'w')
        fp.write(config.dump())
        fp.close()

    # error: revert and end
    if os.system('sudo /usr/lib/siteadm/configtest') != 0:
        revert_config(vhostdir, prefix, tempdir)
        print 'Your config is invalid; exiting!'
        return 1
    
    # everything alright: leave installed, restart server
    remove_backup(tempdir)
    os.system('sudo /usr/lib/siteadm/apachereload')
    
    return 0

def main(args):
    parser = argparse.ArgumentParser(prog=args[0])
    subparsers = parser.add_subparsers()
        
    # reload
    parser_append = subparsers.add_parser(
        'reload',
        help='reconstruct apache vhost config and reload apache')
    parser_append.set_defaults(func=cmd_reload)

    # cdiff
    parser_append = subparsers.add_parser(
        'cdiff',
        help='print a colordiff of applied transformations')
    parser_append.add_argument(
        'file',
        help='file to transform')
    parser_append.set_defaults(func=cmd_cdiff)
    
    namespace = parser.parse_args(args[1:])
    return namespace.func(namespace)
