"""siteadm - per user apache config snippet management tool

Copyright (c) 2009, Florian Wagner <florian@wagner-flo.net>.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

"""

import re

re_line = re.compile(
    '(?P<comment>^[ \t]*(#.*?)?[ \t\n]*$)'
    '|'
    '(?P<blockstart>^[ \t]*\<[^/].*\>[ \t\n]*$)'
    '|'
    '(?P<blockend>^[ \t]*\</.*\>[ \t\n]*$)'
    '|'
    '(?P<nonblock>^[ \t]*[^<].*?[ \t\n]*$)', re.DOTALL)
re_block = re.compile(
    '^(?P<pre>[ \t]*\<)'
    '(?P<name>/?[A-Za-z0-9_]+)'
    '((?P<between>[ \t]+)'
    '(?P<args>.+?))?'
    '(?P<post>[ \t]*\>[ \t\n]*)$', re.DOTALL)
re_nonblock = re.compile(
    '^(?P<pre>[ \t]*)'
    '(?P<name>[A-Za-z0-9_-]+)'
    '(?P<between>[ \t]+)'
    '(?P<args>.*?)'
    '(?P<post>[ \t\n]*)$', re.DOTALL)

class Directive(object):
    def __init__(self, pre, name, between, args, post):
        self.pre = pre
        self.name = name
        self.between = between
        self.post = post
        self._args = None
        self.args = args

    def get_args(self):
        return self._args
    
    def set_args(self, value):
        if value is None:
            return self.del_args()

        if self.between is None:
            self.between = ' '
        
        self._args = value
        
    def del_args(self):
        self._args = None
        self._between = None

    args = property(get_args, set_args, del_args)

    def match(self, args):
        for key,value in args.iteritems():
            if not hasattr(self, key):
                continue

            svalue = getattr(self, key)
            if key == 'name' and value.lower() != svalue.lower():
                return False

            elif key != 'name' and value != svalue:
                return False

        return True

class NonBlockDirective(Directive):
    pass

class BlockStartDirective(Directive):
    pass

class BlockEndDirective(Directive):
    pass

class Comment(object):
    def __init__(self, line):
        self.line = line

    def match(self, args):
        return False

class Block(list):
    def __init__(self, blockstart, blockend, content):
        self._blockstart = blockstart
        self._blockend = blockend
        list.__init__(self, content)

    def get_args(self):
        return self._blockstart.args

    def set_args(self, value):
        self._blockstart.args = value

    def del_args(self):
        del self._blockstart.args

    def get_name(self):
        return self._blockstart.name

    def set_name(self, value):
        self._blockstart.name = value
        self._blockend.name = '/' + value

    args = property(get_args, set_args, del_args)
    name = property(get_name, set_name)

    def match(self, args):
        return self._blockstart.match(args)

    def flatten(self):
        if self._blockstart is not None:
            yield self._blockstart

        for directive in self:
            if isinstance(directive, Block):
                for item in directive.flatten():
                    yield item
            else:
                yield directive

        if self._blockstart is not None:
            yield self._blockend

    def find(self, **args):
        for directive in self:
            if directive.match(args):
                yield directive

    def find_recursive(self, **args):
        for directive in self:
            if directive.match(args):
                yield directive

            if isinstance(directive, Block):
                for item in directive.find_recursive(**args):
                    yield item

    def append_nonblock(self, **args):
        args.setdefault(None)
        
        pre = args.get('pre')
        if pre is None:
            for item in reversed(self):
                if isinstance(item, Directive):
                    pre = item.pre
                    break

        self.append(NonBlockDirective(pre,
                                      args.get('name'),
                                      args.get('between'),
                                      args.get('args'),
                                      args.get('post', '\n')))

class Config(Block):
    def __init__(self, filename):
        self._filename = filename
        Block.__init__(self, None, None, self._parse())

    def match(self, args):
        return False

    def _parse_directive(self, directive):
        if directive.group('comment') is not None:
            return Comment(directive.group('comment'))

        elif directive.group('nonblock') is not None:
            return NonBlockDirective(
                **re_nonblock.match(directive.group('nonblock')).groupdict())

        elif directive.group('blockstart') is not None:
            return BlockStartDirective(
                **re_block.match(directive.group('blockstart')).groupdict())

        elif directive.group('blockend') is not None:
            return BlockEndDirective(
                **re_block.match(directive.group('blockend')).groupdict())

        return None

    def _parse_block(self, blockstart, directives):
        blockcontent = []

        for directive in directives:
            directive = self._parse_directive(directive)

            if directive is None:
                raise Exception

            if isinstance(directive, BlockStartDirective):
                blockcontent.append(
                    self._parse_block(directive, directives))

            elif isinstance(directive, BlockEndDirective):
                return Block(blockstart, directive, blockcontent)

            else:
                blockcontent.append(directive)

        return blockcontent

    def _parse(self):
        fp = file(self._filename)

        def directive_generator():
            fulldirective = None

            for line in fp:
                if line.endswith('\\\n'):
                    if fulldirective is None:
                        fulldirective = line
                    else:
                        fulldirective = fulldirective + line
                else:
                    if fulldirective is not None:
                        yield re_line.match(fulldirective + line)
                        fulldirective = None
                    else:
                        yield re_line.match(line)

        return self._parse_block(None, directive_generator())

    def dump_lines(self):
        def emtpy_when_none(item):
            return item != None and item or ''

        for item in self.flatten():
            if isinstance(item, Comment):
                yield item.line

            else:
                yield ''.join([emtpy_when_none(item.pre),
                               item.name,
                               emtpy_when_none(item.between),
                               emtpy_when_none(item.args),
                               item.post])

    def dump(self):
        return ''.join(self.dump_lines())
