# -*-  encoding: utf-8
"""
 :copyright: (c) 2009 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""

import cgi
from itertools import takewhile

def chunkify(stream, size, chunksize):
    while size > 0:
        chunk = stream.read(min(size, chunksize))
        size -= len(chunk)
        yield chunk
               

class Parser(object):

    linebreak = '\r\n'
    chunksize = 1024 * 100

    def __init__(self, stream, size, boundary):
        self.iterator = chunkify(stream, size, self.chunksize)        
        
        self.size = size
        self.bytesread = 0

        self.boundary = boundary
        self.buffer = ''
        self.mode = self.parse_start

    def __iter__(self):
        return self

    def next(self):
        while True:
            if not self.hasnext():
                raise StopIteration
            maxsize = min(self.chunksize, self.size - self.bytesread)
            if maxsize:
                chunk = self.iterator.next()
                self.bytesread += len(chunk)
                self.buffer += chunk
                
            token = self.mode()
            if token:
                return token                               
                
    def hasnext(self):
        return self.mode is not None

    def ensure_buffer_size(self, size):
        return len(self.buffer) >= size or self.size == self.bytesread


    def parse_start(self):
        """ parser mode
        consume --boundary -> after_boundary
        """
        dashboundary = '--' + self.boundary
        if not self.ensure_buffer_size(len(dashboundary)):
            return
        if not self.buffer.startswith(dashboundary):
            raise Exception("expected '--' + boundary")
        self.buffer = self.buffer[len(dashboundary):]
        self.mode = self.parse_after_boundary


    def parse_after_boundary(self):
        """ parser mode
        consume linebreak -> headers
        consume --        -> end
        """
        if not self.ensure_buffer_size(2):
            return
        if self.buffer.startswith('--'):
            self.mode = None
        elif self.buffer.startswith(self.linebreak):
            self.buffer = self.buffer[len('--'):]
            self.mode = self.parse_headers      
            return ('field_start', None)
        else:
            raise Exception("expected -- or CRLF after boundary")    


    def parse_headers(self):
        """ parser mode
        consume line       -> yield header
        consume empty line -> chunks
        """
        if self.buffer.startswith(self.linebreak):
            self.buffer = self.buffer[len(self.linebreak):]
            self.mode = self.parse_chunks
            return ('data_start', None)
        elif self.linebreak in self.buffer:
            header, self.buffer = self.buffer.split(self.linebreak, 1)
            key, value = header.split(':', 1)
            return ('header', (key.lower(), value))


    def parse_chunks(self):
        """ parser mode
        consume anything not containing --boundary -> yield chunk
        consume --boundary                         -> after_boundary
        """
        dashboundary = self.linebreak + '--' + self.boundary
        if not self.ensure_buffer_size(len(dashboundary) * 2):
            return
        if dashboundary in self.buffer:
            chunk, self.buffer = self.buffer.split(dashboundary, 1)
            self.mode = self.parse_after_boundary
        else:
            chunk = self.buffer[:-len(dashboundary)]
            self.buffer = self.buffer[-len(dashboundary):]
        return ('chunk', chunk)

    def max_payload_size_left(self):
        """ the maximum size of payload left
        """
        bytes_left = self.size - self.bytesread + len(self.buffer)
        return bytes_left - len(2 * self.linebreak + 2 * '--' + self.boundary)



class Form(object):

    def __init__(self, stream, size, boundary):
        self.parser = Parser(stream, size, boundary)

    def __iter__(self):
        assert self.parser.next() == ('field_start', None)
        while self.parser.hasnext():
            field = Field(self.parser)
            yield field
            field.exhaust()
                       
            
class Field(object):
    """
    
    maxsize - the maximum possible size, assuming no following fields
    iterator - the iterator to iterate the content
    """
    def __init__(self, parser):
        self.parser = parser
        self.headers = dict([header for _, header 
                              in takewhile(lambda t: t[0] == 'header', parser)])
        self.maxsize = parser.max_payload_size_left()
        self._content = None
                              
        
    def __iter__(self):
        if self.parser is None:
            return
        for _, chunk in takewhile(lambda t: t[0] == 'chunk', self.parser):
            yield chunk
        self.parser = None

    def exhaust(self):
        for _ in self:
            pass
            
    @property
    def content(self):
        if self._content is None:
            self._content = ''.join(self)
        return self._content
                        
    @property
    def name(self):
        """ the name of the field as given in the form
        """
        value, args = cgi.parse_header(self.headers['content-disposition'])
        return args['name']

    @property
    def filename(self):        
        """ the name of the uploaded file, if this is a filefield
        """
        value, args = cgi.parse_header(self.headers['content-disposition'])
        return args.get('filename', None)               


