# -*-  encoding: utf-8
"""
 :copyright: (c) 2009 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""

from werkzeug.utils import import_string

class FileSlot(object):

    @classmethod
    def create(cls, model, filedb, user_id):
        """ creates a new fileslot for the given user
        """
        fileslot_id = filedb.create_fileslot(user_id)
        return cls.fetch(model, filedb, fileslot_id)
        
    @classmethod
    def fetch(cls, model, filedb, fileslot_id):
        """ fetches a fileslot from the database 
        """
        try:
            user_id, filename, size, error, created_at = filedb.get_fileslot_info(fileslot_id)
            return cls(model, filedb, fileslot_id, user_id, filename, size, error, created_at) 
        except filedb.NotFound:
            raise NotFound("fileslot not found")    
    
    """
    public fields
    
    - id
    - filename
    - size    
    """
    def __init__(self, model, filedb, id_, user_id, filename, size, error, created_at):
        self.model = model
        self.filedb = filedb
        self.id = id_
        self.filename = filename
        self.size = size
        self.user_id = user_id
        self.error = error
        self.created_at = created_at

    def upload_started(self):
        """ returns true if an upload has started, false otherwise
        """
        return self.filename is not None
        

    def progress(self):
        """ returns the progress, as a float between 0.0 and 1.0
        """
        if not self.upload_started():
            return 0.0
        else:
            return self.filedb.progress_for(self.id)
                        
                                
    def store_file(self, filename, size, iterator):
        """ stores the file with the given filename, using the data from the
        iterator as content          
        """        
        if self.upload_started():
            raise UploadException("a file is already uploaded")
        elif size > self.get_user().max_file_size:
            errormsg = 'file too big'
            self.filedb.store_error(self.id, errormsg)
            raise UploadException(errormsg)
        else:
            self._clear_for(size)
            self.filename = filename
            self.size = size            
            self.filedb.store_file_data(self.id, filename, size, iterator)            
            
    def _clear_for(self, size):
        """ clears files of the user with the selected cleaning strategy
            to allow an upload of at least size
        """
        max_size = self.get_user().max_total_size - size
        self.cleaning_strategy(self.filedb, self.user_id, max_size)
        
        
    def cleaning_strategy(self, filedb, user_id, max_size):
        """ cleans the oldest files first
        """
        lst = []
        for fileslot_id in filedb.get_fileslot_ids_for_user(user_id):
            _, _, size, _, created_at = filedb.get_fileslot_info(fileslot_id)
            if size:
                lst.append((created_at, size, fileslot_id))

        lst.sort()
        total_size = reduce(lambda size, tpl: tpl[1] + size, lst, 0)
                
        for _, size, fileslot_id in lst:
            if total_size <= max_size:
                break
            filedb.free_fileslot(fileslot_id)
            total_size -= size


    def get_file(self):
        """ returns a contentiterator
        """
        if not self.upload_started():
            raise Exception("someone should start upload a file first")
        else:            
            return self.filedb.get_file_data(self.id)
        

    def get_user(self):
        """ returns the user who create this fileslot
        """        
        return self.model.get_user_by_id(self.user_id)


class User(object):
    """
    public fields
    
    - login
    - max_file_size
    - max_total_size
    """    
    def __init__(self, model, login, max_file_size, max_total_size):
        self.model = model
        self.login = login
        self.max_file_size = max_file_size
        self.max_total_size = max_total_size

    def create_fileslot(self):
        return self.model._create_fileslot(self.login)


class NotFound(Exception):
    pass        
    
class UploadException(Exception):
    pass                        
    
    
class Model(object):
    get_plugin_factory = staticmethod(import_string)    
    fileslot_factory = FileSlot
    
    NotFound = NotFound
    UploadException = UploadException
                
    def __init__(self, config):                
        self.userdb = self._create_plugin(config['userdb'])
        self.filedb = self._create_plugin(config['filedb'])        
        
    def _create_plugin(self, config):
        factory = self.get_plugin_factory(config['plugin'])
        return factory(self, config)        
                
    def _create_fileslot(self, user_id):
        return self.fileslot_factory.create(self, self.filedb, user_id)
                                 
    def get_fileslot(self, fileslot_id):
        return self.fileslot_factory.fetch(self, self.filedb, fileslot_id)
            
    def get_anonymous_user(self):
        return self.userdb.get_anonymous_user()
        
    def get_user_by_id(self, user_id):
        return self.get_user_by_login(user_id)

    def get_user_by_login(self, login):
        return self.userdb.get_user_by_login(login)
        
    def get_user_by_credentials(self, login, password):
        return self.userdb.get_user_by_credentials(login, password)

