# -*-  encoding: utf-8
"""
 :copyright: (c) 2009 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""

from glob import glob
import os
import os.path
import pickle
import random
from datetime import datetime
import time
import string

class NotFound(Exception):
    pass


class FsFileDb(object):

    maxid = 10 ** 20
    chunksize = 1024 * 10
    waitinterval = 1
    
    pool = string.digits + string.uppercase       
    open_file = open
    remove_file = os.remove
    
    NotFound = NotFound
    
    now = datetime.now    

    def __init__(self, model, config):
        self.model = model
        self.prefix = config.get('storage_path', '/tmp')

    
    def create_fileslot(self, user_id):
        """ creates a fileslot, with the given user_id as owner, and returns
        the id of the fileslot
        """
        fileslot_id = self._encode_id(random.randint(0, self.maxid))
                
        infofile = InfoFile(self._get_filename(fileslot_id, 'info'))
        infofile.slot_id = fileslot_id
        infofile.user_id = user_id
        infofile.created_at = self.now()
        infofile.store()                
        
        return fileslot_id
        
    def free_fileslot(self, fileslot_id):
        self.remove_file(self._get_filename(fileslot_id, 'data'))
        self.remove_file(self._get_filename(fileslot_id, 'info'))
        
    def get_fileslot_ids_for_user(self, user_id):
        """
        """
        for filename in glob(os.path.join(self.prefix, '*.info')):
            infofile = InfoFile(filename)
            infofile.load()
            if infofile.user_id == user_id:
                yield infofile.slot_id
                    
        
    def get_fileslot_info(self, fileslot_id):
        """ returns a (user_id, filename, size, error, created_at)-tuple for the given file_id
        """
        infofile = self._get_infofile(fileslot_id)
        return (infofile.user_id, infofile.filename, infofile.size, 
                                            infofile.error, infofile.created_at)
        
    def store_file_data(self, fileslot_id, filename, size, iterator):
        """ read the content out of the iterator and save it incrementally.
        """ 
        infofile = self._get_infofile(fileslot_id)        
        infofile.filename = filename
        infofile.size = size
        infofile.store()
       
        fp = self.open_file(self._get_filename(fileslot_id, 'data'), 'wb')
        try:
            for chunk in iterator:
                fp.write(chunk)
        finally:
            fp.close()
            
    def store_error(self, fileslot_id, error):
        """ stores an error that happened while handling the file
        """
        infofile = self._get_infofile(fileslot_id)
        infofile.error = error
        infofile.store()            

    def get_file_data(self, fileslot_id):
        infofile = self._get_infofile(fileslot_id)        
        fp = self.open_file(self._get_filename(fileslot_id, 'data'), 'rb')
        size = infofile.size
        # try/finally would be better, but no python2.4 support in generators
        # yay, be friendly to the ancients (aka debian)

        while size > 0:
            chunk = fp.read(min(self.chunksize, size))
            if chunk:
                size -= len(chunk)
                yield chunk
            else:
                time.sleep(self.waitinterval)

        fp.close()


    def progress_for(self, fileslot_id):
        """ returns the upload progress of a file, as a value between 0.0 and 1.0
        """
        infofile = self._get_infofile(fileslot_id)        
        return 1.0 * os.stat(self._get_filename(fileslot_id, 'data')).st_size / infofile.size        
        
    def _encode_id(self, number):
        digits = []
        while True:
            number, digit = divmod(number, len(self.pool))
            digits.append(self.pool[digit])            
            if number == 0:
                return ''.join(digits[::-1])

    def _get_filename(self, fileslot_id, ext):
        return os.path.join(self.prefix, '%s.%s' % (fileslot_id, ext))                        
        
    def _get_infofile(self, fileslot_id):
        infofile = InfoFile(self._get_filename(fileslot_id, 'info'))
        try:
            infofile.load()
            return infofile
        except IOError:
            raise self.NotFound("no such fileslot")



class InfoFile(object):
    managed_attributes = ('slot_id', 'filename', 'size', 'user_id', 'error',
                                                                   'created_at')

    def __init__(self, path):
        self.path = path
        for name in self.managed_attributes:
            setattr(self, name, None)
        
    def store(self):
        fp = open(self.path, 'wb')
        try:
            data = dict((name, getattr(self, name)) 
                                            for name in self.managed_attributes)                
            pickle.dump(data, fp)
        finally:
            fp.close()    
        
    def load(self):
        fp = open(self.path, 'rb')
        try:
            data = pickle.load(fp)
            for name in self.managed_attributes:
                setattr(self, name, data[name])
        finally:
            fp.close()        

