# -*-  encoding: utf-8
"""
 :copyright: (c) 2009 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""

from simplejson import dumps
import mako
import mako.lookup


from werkzeug.wrappers import BaseResponse
from werkzeug.routing import redirect

class Response(BaseResponse):
    charset = 'utf-8'    

class JsonResponse(BaseResponse):
    default_mimetype = 'text/json'


class FileResponse(BaseResponse):       
    def __init__(self, iterator, size, filename, mimetype=None):
        headers = [ ('Content-Transfer-Encoding', 'binary') ]
        headers.append(('Content-Length', size))
        headers.append(('Content-Disposition', 
                                        'attachment; filename="%s"' % filename))
        BaseResponse.__init__(self, response=iterator, mimetype=mimetype,
                                                                headers=headers)
                                                                
class MakoResponse(BaseResponse):
    default_mimetype = 'text/html'
    
    def __init__(self, lookup, template, status, args):
        try:            
            tpl = lookup.get_template(template)
            data = tpl.render_unicode(**args)
            BaseResponse.__init__(self, response=data, status=status)
        except:
            data = mako.exceptions.html_error_template().render_unicode()
            raise TemplateException(data)
            

class TemplateException(Exception, Response):
    default_mimetype = 'text/html'

    def __init__(self, msg):
        Exception.__init__(self, msg)
        Response.__init__(self, response=msg, status=500)
                                                                

class View(object):
    
    def __init__(self, templatepath):
        self.lookup = mako.lookup.TemplateLookup(templatepath)
            
    def render_template(self, template, status, args):
        return MakoResponse(self.lookup, template, status, args)

    def redirect_to(self, url):
        return redirect(url)

    def render_file_download(self, iterator, size, filename, mimetype=None):
        return FileResponse(iterator, size, filename, mimetype)

    def json_response(self, data):        
        return JsonResponse(dumps(data))
        
        
def template_exception_middleware(app):
    """ middleware that catches template exceptions and formats them nicely
    """
    def wrapper(*args):
        try:
            return app(*args)
        except TemplateException, exc:
            return exc(*args)
    return wrapper

