# -*-  encoding: utf-8
"""
 :copyright: (c) 2009 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""

import cgi
from lib import Form

import helper

class Base(object):

    def __init__(self, request, url_adapter, model, view):
        self.request = request
        self.url_adapter = url_adapter
        self.model = model
        self.view = view        
                
    def url_for(self, endpoint, force_external=False, **values):
        return self.url_adapter.build(endpoint, values, 'GET', force_external)

    def render_template(self, template, status=200, **kwargs):
        kwargs.update({
            'url_for' : self.url_for,
            'current_user' : self.current_user,
            'as_filesize' : helper.as_filesize,
        })
        return self.view.render_template(template, status, kwargs)
        
    def render_json(self, obj):
        return self.view.json_response(obj)
        
    def render_file_download(self, iterator, size, filename, mimetype=None):
        return self.view.render_file_download(iterator, size, filename, mimetype)
        
    def redirect_to(self, endpoint, **values):
        return self.view.redirect_to(self.url_for(endpoint, **values))        
        
    def get_current_user(self):
        login = self.request.session.get('user_login', None)
        if login is None:
            return self.model.get_anonymous_user()
        else:
            return self.model.get_user_by_login(login)
            
    def set_current_user(self, user):
        self.request.session['user_login'] = user.login
        
    current_user = property(get_current_user, set_current_user)

    @classmethod
    def with_fileslot(cls, func):
        def wrapper(self):
            try:
                self.fileslot = self.model.get_fileslot(
                                                 self.request.params['fileslot_id'])
            except self.model.NotFound:
                return self.render_template('not_found')
            else:
                return func(self)
        return wrapper



class TemplateController(Base):
    """ Base Controller for Controller that just render templates
    """
    template = None
    args = {}
    def run(self):
        return self.render_template(self.template, **self.args)
    

class Info(TemplateController):
    template = 'info.html.mako'

class Credits(TemplateController):
    template = 'credits.html.mako'

class License(TemplateController):
    template = 'license.html.mako'

class Profile(TemplateController):
    template = 'profile.html.mako'

class LoginGet(TemplateController):
    template = 'login.html.mako'
    args = { 'error' : None }
    
class NotFound(TemplateController):
    args = { 'status' : 404 }
    template = 'not_found.html.mako'
        

class LoginPost(Base):
    def run(self):
        try:
            login = self.request.params.get('login', '')
            passwd = self.request.params.get('password', '')
            self.current_user = self.model.get_user_by_credentials(login, passwd)
            return self.redirect_to('Profile')
        except self.model.NotFound, exc:
            return self.render_template('login.html.mako', error=unicode(exc))
                        
                        
class CreateFileslot(Base):
    def run(self):
        fileslot = self.current_user.create_fileslot()
        if self.request.expects_json():
            return self.render_json(self.fileslot_to_json(fileslot))
        else:
            return self.redirect_to('UploadGet', fileslot_id=fileslot.id)

    def fileslot_to_json(self, fileslot):
        return {
            'fileslot_id' : fileslot.id,
            'urls' : {
                'upload' : self.url_for('Upload'),
                'preview' : self.url_for('Preview', fileslot_id=fileslot.id),
                'download' : self.url_for('Download', fileslot_id=fileslot.id),
            },
        }


class UploadGet(Base):  

    @Base.with_fileslot
    def run(self):
        if not self.fileslot.upload_started():           
            return self.render_template('upload.html.mako', 
                                                         fileslot=self.fileslot)
        elif self.fileslot.progress() < 1.0:
            return self.render_template('upload_already_started.html.mako', 
                                                         fileslot=self.fileslot)
        else:
            return self.redirect_to('Preview', fileslot_id=self.fileslot.id)                


class UploadPost(Base):
    formfactory = Form

    #@Base.with_fileslot
    def run(self):
        try:
            fileslot = self.model.get_fileslot(
                                         self.request.url_values['fileslot_id'])

            for field in self.create_formparser():
                if field.name != 'file':
                    raise Exception("expected 'file' field only, got '%s'" 
                                                                   % field.name)
                elif fileslot.upload_started():
                    raise Exception("only one 'file' field expected")
                else:
                    fileslot.store_file(field.filename, field.maxsize, field)

            return self.render_template('info.html.mako')
            
        except self.model.UploadException, exc:
            return self.render_template('upload_error.html.mako', status=403,
                                                             error=unicode(exc))


    def create_formparser(self):
        value, args = cgi.parse_header(self.request.headers['Content-Type'])
        if value != 'multipart/form-data':
            raise Exception('expected the uploads Content-Type to '
                                   'be "multipart/form-data", got "%s"' % value)
                                   
        boundary = args['boundary']
        size = int(self.request.headers['Content-Length'])
        stream = self.request.input_stream
        
        return self.formfactory(stream, size, boundary)
                

class Preview(Base):

    @Base.with_fileslot
    def run(self):
        if self.request.expects_json():
            return self.render_json(self.fileslot_to_json(self.fileslot))
        else:
            if not self.fileslot.error:
                return self.render_template('preview.html.mako', fileslot=self.fileslot)
            else:
                return self.render_template('upload_error.html.mako', fileslot=self.fileslot)
            
    def fileslot_to_json(self, fileslot):
        return {
            'filename' : fileslot.filename,
            'size'     : fileslot.size,
            'progress' : fileslot.progress() * 100,
            'error'    : fileslot.error,
            #'download' : self.url_for('Download', fileslot_id=fileslot.id)
        }

    
class Download(Base):
    @Base.with_fileslot
    def run(self):
        return self.render_file_download(self.fileslot.get_file(), 
                                     self.fileslot.size, self.fileslot.filename)
