# -*-  encoding: utf-8
"""
 :copyright: (c) 2009 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""

from werkzeug.utils import CombinedMultiDict
from werkzeug.wrappers import BaseRequest


class Request(BaseRequest):
    charset = 'utf-8'

    def __init__(self, environ, url_values):
        BaseRequest.__init__(self, environ)
        self.url_values = url_values

    @property
    def params(self):
        """ returns a combined multidict with
                - parsed values form url
                - values from POST
                - values from GET
                - values from posted files
        """
        return CombinedMultiDict([self.url_values, self.form, self.args, 
                                                                    self.files])
            
    @property                                                        
    def session(self):
        return self.environ['securecookie']
                                                                    
    def expects_json(self):
        return self.headers.get('accept', None) == 'application/json'





from werkzeug.utils import cookie_date, parse_cookie, dump_cookie
from werkzeug.contrib.securecookie import SecureCookie


class SecureCookieMiddleware(object):

    def __init__(self, app, secret_key, environ_key='securecookie', 
                                   cookie_name='session_data', charset='utf-8'):
        self.app = app
        self.secret_key = secret_key
        self.environ_key = environ_key
        self.cookie_name = cookie_name
        self.charset = charset
        
    def get_secure_cookie(self, environ):
        cookies = parse_cookie(environ, self.charset)        
        data = cookies.get(self.cookie_name)
        if not data:
            return SecureCookie(secret_key=self.secret_key)
        else:
            return SecureCookie.unserialize(data, self.secret_key)        
            
            
    def set_secure_cookie(self, environ, headers):
        sc = environ[self.environ_key]
        if sc.should_save:
            data = sc.serialize()            
            data = dump_cookie(self.cookie_name, data, path='/', charset=self.charset)
            headers.append( ('Set-Cookie', data) )                    
        

    def __call__(self, environ, start_response):    
        environ[self.environ_key] = self.get_secure_cookie(environ)
        
        def start_response_wrapper(status, headers):
            self.set_secure_cookie(environ, headers)
            return start_response(status, headers)
    
        return self.app(environ, start_response_wrapper)
