import pathfix

from skarabaeus_server import controller
import testhelper as testing
from testhelper import before_decorator


class FakeUrl(testing.Opaque):
    pass


class ControllerTest(object):

    controller_factory = None

    def setup_method(self, method):    
        self.setup_model()
        self.setup_controller()
        
    def setup_model(self):
        self.file = testing.Object(
            id = 232323,
            size = 10,
            filename = 'test.txt',
            error = None,
            get_file = testing.Method(returns=range(10)),
            progress = testing.Method(returns=0.235),
            store_file = testing.Method(returns=None),            
        )
        
        self.user = testing.Object(
            create_fileslot = testing.Method(returns=self.file)                
        )

        self.model = testing.Object(
            get_fileslot = testing.Method(returns=self.file)
        )
    
    def setup_controller(self):
        self.url_adapter = testing.Object()
        
        self.request = testing.Object(
            input_stream = testing.Object(),
            headers = {},
            params = {},
            expects_json = testing.Method(returns=False),        
        )                                
        
        self.controller = self.controller_factory(self.request, 
                                             self.url_adapter, self.model, None)

        self.controller.redirect_to = testing.Method(returns='redirect')
        self.controller.render_template = testing.Method(returns='render_template')
        self.controller.render_json = testing.Method(returns='render_json')
        self.controller.render_file_download = testing.Method(returns='render_file_download')
        
        self.controller.url_for = FakeUrl
        
        type(self.controller).current_user = self.user        
                        
    def must_render_template(self, template, **kwargs):
        """ asserts, that the template with the given kwargs is rendered and returned.
        """
        assert self.controller.run() == 'render_template'
        self.controller.render_template.must_be_called_with(template, **kwargs)
        
    def must_render_json(self, data):
        """ asserts, that json with the given data is rendered and returned
        """
        self.request.expects_json = testing.Method(returns=True)
        assert self.controller.run() == 'render_json'
        self.controller.render_json.must_be_called_with(data)
        
    def must_redirect_to(self, endpoint, **values):
        """ asserts, that a redirection is returned
        """
        assert self.controller.run() == 'redirect'
        self.controller.redirect_to.must_be_called_with(endpoint, **values)
        
    def must_download_file(self, content, size, filename):
        """ asserts, that a file is provided for download
        """
        assert self.controller.run() == 'render_file_download'
        self.controller.render_file_download.must_be_called_with(content, size, filename)

                        
def run_controller(as_json=False):
    @before_decorator
    def decorator(self):
        self.request.expects_json = testing.Method(returns=as_json)
        self.result = self.controller.run()

    return decorator
        
        
def with_file(progress=None, upload_started=None):
    """ decorator to set up a file fixture
    """
    @before_decorator
    def decorator(self):        
        self.file.upload_started = testing.Method(returns=upload_started)
        self.file.progress = testing.Method(returns=progress)

    return decorator


class TestSimpleTemplateController(object):
    def must_render_controller_template(self, factory, template, **args):
        """ asserts, that a controller created from the factory renders the template
        """
        controller = factory(None, None, None, None)
        controller.render_template = testing.Method(returns='result')
        assert controller.run() == 'result'
        controller.render_template.must_be_called_with(template, **args)
        
    def test_info(self):
        self.must_render_controller_template(controller.Info, 'info.html.mako')
        
    def test_credits(self):
        self.must_render_controller_template(controller.Credits, 'credits.html.mako')

    def test_profile(self):
        self.must_render_controller_template(controller.Profile, 'profile.html.mako')

    def test_login(self):
        self.must_render_controller_template(controller.LoginGet, 'login.html.mako', error=None)

    def test_not_found(self):
        self.must_render_controller_template(controller.NotFound, 'not_found.html.mako', status=404)


class TestLoginPost(ControllerTest):
    controller_factory = controller.LoginPost    
    
    
class TestCreateFileslotController(ControllerTest):
    controller_factory = controller.CreateFileslot
    
    @run_controller()
    def test_creates_fileslot(self):
        self.user.create_fileslot.must_be_called_without_args()    
    
    def test_redirect(self):
        self.must_redirect_to('UploadGet', fileslot_id=self.file.id)

    def test_renders_json(self):
        self.must_render_json({
           'fileslot_id' : self.file.id,
            'urls' : {
                'upload' : FakeUrl('Upload'),
                'preview' : FakeUrl('Preview', fileslot_id=self.file.id),
                'download' : FakeUrl('Download', fileslot_id=self.file.id),
            }
        })
        
       
class TestUploadGetController(ControllerTest):
    controller_factory = controller.UploadGet
    
    def setup_method(self, method):
        super(TestUploadGetController, self).setup_method(method)            
        self.request.params['fileslot_id'] = 23235
        
    @with_file(upload_started=False, progress=0.0)
    def test_rendering_for_empty_fileslot(self):
        self.must_render_template('upload.html.mako', fileslot=self.file)

    @with_file(upload_started=True, progress=0.23)
    def test_rendering_while_uploading(self):
        self.must_render_template('upload_already_started.html.mako', fileslot=self.file)
        
    @with_file(upload_started=True, progress=1.0)
    def test_redirect_when_uploaded(self):
        self.must_redirect_to('Preview', fileslot_id=self.file.id)
        
    @with_file(upload_started=False, progress=0.0)
    @run_controller()
    def test_creates_id(self):        
        self.model.get_fileslot.must_be_called_with(23235)
        
        
class TestUploadPostController(ControllerTest):
    controller_factory = controller.UploadPost
    
    def setup_method(self, method):
        super(TestUploadPostController, self).setup_method(method)
        self.request.url_values = {
            'fileslot_id' : 5,   
        }
        self.request.headers = {
            'Content-Type' : 'multipart/form-data; boundary=--23',
            'Content-Length' : '23',
        }
        self.file_field = testing.Object(
            name='file',
            filename='principia discordiae.txt',
            maxsize=23235,
        )        
        self.form = [self.file_field]
        self.controller.formfactory = testing.Method(returns=self.form)
        self.file.upload_started = testing.Method(returns=False)

    
    def test_rendering(self):
        self.must_render_template('info.html.mako')

    @run_controller()               
    def test_uses_formfactory(self):
        self.controller.formfactory.must_be_called_with(
                                          self.request.input_stream, 23, '--23')
        
    @run_controller()
    def test_stores_file(self):
        self.file.store_file.must_be_called_with(self.file_field.filename, 
                                       self.file_field.maxsize, self.file_field)
    
        
class TestDownloadController(ControllerTest):
    controller_factory = controller.Download
    
    def setup_method(self, method):
        super(TestDownloadController, self).setup_method(method)
        self.request.params['fileslot_id'] = 23
    
    @run_controller()
    def test_queries_model(self):
        self.model.get_fileslot.must_be_called_with(self.request.params['fileslot_id'])
            
    def test_downloads_file(self):
        self.must_download_file(self.file.get_file(), self.file.size, self.file.filename)
                     
                                                                                    
class TestPreviewController(ControllerTest):

    controller_factory = controller.Preview
    
    def setup_method(self, method):
        super(TestPreviewController, self).setup_method(method)                
        self.request.params['fileslot_id'] = 23
    
    @run_controller()
    def test_queries_model(self):
        self.model.get_fileslot.must_be_called_with(self.request.params['fileslot_id'])
        
    def test_renders_template(self):
        self.must_render_template('preview.html.mako', fileslot=self.file)
        
    def test_renders_error_page(self):
        self.file.error = 'omfg!'
        self.must_render_template('upload_error.html.mako', fileslot=self.file)
    
    def test_renders_json(self):
        self.must_render_json({
            'filename' : self.file.filename,
            'size' : self.file.size,
            'progress' : self.file.progress() * 100,
            'error' : None,
        })

