import pathfix
import py.test
import testhelper as testing

from skarabaeus_server.model import FileSlot, User, Model
from skarabaeus_server import model

class FakeException(Exception):
    pass


class TestModel(object):
    
    def setup_method(self, method):       
        class CoopModel(Model):
            get_plugin_factory = self.get_plugin_factory
            fileslot_factory = testing.Opaque

        self.filedb = testing.Object()
        self.userdb = testing.Object()        
        
        self.model = CoopModel({
            'userdb' : { 'plugin' : 'plugin-userdb' },
            'filedb' : { 'plugin' : 'plugin-filedb' },
        })        
    
    def get_plugin_factory(self, name):
        if name == 'plugin-userdb':
            return lambda model, config: self.userdb
        elif name == 'plugin-filedb':
            return lambda model, config: self.filedb
            
    def must_delegate(self, plugin, methodname, prefix, *args):    
        """ asserts, that model.<methodname>(*args) is delegated to 
        plugin.<methodname>(*args)
        """
        targetfunc = testing.Method(returns=methodname)
        setattr(plugin, methodname, targetfunc)            
        modelfunc = getattr(self.model, prefix + methodname)        
        assert modelfunc(*args) == methodname
        targetfunc.must_be_called_with(*args)                    
                

    def test_delegates_get_anonymous_user(self):
        self.must_delegate(self.userdb, 'get_anonymous_user', '')

    def test_delegates_get_user_by_login(self):
        self.must_delegate(self.userdb, 'get_user_by_login', '', 'discordia')

    def test_delegates_get_user_by_login(self):
        self.must_delegate(self.userdb, 'get_user_by_credentials', '', 'discordia', '23')
                
    def test_get_fileslot(self):
        self.model.fileslot_factory.fetch = testing.Method(returns='fileslot')
        assert self.model.get_fileslot('fileslot_id') == 'fileslot'
        self.model.fileslot_factory.fetch.must_be_called_with(self.model, self.filedb, 'fileslot_id')

        #self.filedb.get_fileslot_info = testing.Method(
        #                     returns=('user_id', 'filename.txt', 2323, 'error'))
        #                     
        #fileslot = self.model.get_fileslot('fileslot_id')
       # 
       # assert fileslot == testing.Opaque(self.model, self.filedb,
       #                 'fileslot_id', 'user_id', 'filename.txt', 2323, 'error')


    def test_create_fileslot(self):
        self.model.fileslot_factory.create = testing.Method(returns='fileslot')
        
        assert self.model._create_fileslot('user_id') == 'fileslot'
        self.model.fileslot_factory.create.must_be_called_with(self.model, self.filedb, 'user_id')
                            

class TestUser(object):

    def setup_method(self, method):
        self.model = testing.Object(
            _create_fileslot=testing.Method(returns='fileslot')
        )

    def test_creation(self):    
        user = User(self.model, 'discordia', 5, 23)
        assert user.login == 'discordia'
        assert user.max_file_size == 5
        assert user.max_total_size == 23
        
    def test_create_slot(self):
        user = User(self.model, 'discordia', 5, 23)
        fileslot = user.create_fileslot()
        assert fileslot == 'fileslot'
        self.model._create_fileslot.must_be_called_with('discordia')        
        

class FileSlotBase(object):
    def setup_mocks(self):
        self.user = User(None, 'discordia', 5, 23)
        self.model = testing.Object(
            get_user_by_id=testing.Method(returns=self.user),            
        )                
        
        self.filedb = testing.Object(
            store_file_data=testing.Method(returns=None),
            store_error=testing.Method(returns=None),                   
            get_fileslot_info = testing.Method(returns=('user_id', 'filename', 100, 'error', 'now')),
            UploadException=Model.UploadException,
            NotFound = FakeException,
        )        
        self.fileslot = FileSlot(self.model, self.filedb, 23, 'user_id', None, None, None, 'now')
        self.fileslot.cleaning_strategy = testing.Method(returns=None)    


class TestFileSlot(FileSlotBase):
    def setup_method(self, method):
        self.setup_mocks()
        
    def test_init(self):
        fileslot = FileSlot(self.model, self.filedb, 23, 'user_id', 'filename', 
                                                                   100, 'error', 'now')
        assert fileslot.id == 23
        assert fileslot.filename == 'filename'
        assert fileslot.size == 100
        assert fileslot.error == 'error'
        assert fileslot.created_at == 'now'
        
    def test_fetch(self):
        fileslot = FileSlot.fetch(self.model, self.filedb, 23)
        assert fileslot.id == 23
        assert fileslot.filename == 'filename'
        assert fileslot.size == 100
        assert fileslot.error == 'error'
        assert fileslot.created_at == 'now'
        
    def test_fetch_unknown(self):
        self.filedb.get_fileslot_info = testing.Method(raises=self.filedb.NotFound)
        py.test.raises(model.NotFound, FileSlot.fetch, self.model, self.filedb, 666)        
        
    def test_get_user(self):
        self.model.get_user_by_id = testing.Method(returns='user')        
        assert self.fileslot.get_user() == 'user'
        self.model.get_user_by_id.must_be_called_with('user_id')
                        
    def test_upload_started(self):
        assert not self.fileslot.upload_started()
        self.fileslot.store_file('filename', 5, range(5))
        assert self.fileslot.upload_started()
        
    def test_progress(self):
        self.filedb.progress_for = testing.Method(returns=2.3)
    
        assert self.fileslot.progress() == 0.0
        self.fileslot.store_file('filename', 5, range(5))
        assert self.fileslot.progress() == 2.3        
        
    def test_get_file(self):
        self.filedb.get_file_data = testing.Method(returns='content')
        py.test.raises(Exception, self.fileslot.get_file)
        self.fileslot.store_file('filename', 5, range(5))
        
        assert self.fileslot.get_file() == 'content'
        self.filedb.get_file_data.must_be_called_with(23)        
        

class TestSuccessfulFileSlotStorage(FileSlotBase):

    def setup_method(self, method):
        self.setup_mocks()

    def test_delegates_to_filedb(self):
        self.fileslot.store_file('filename', 5, range(5))        
        self.filedb.store_file_data.must_be_called_with(23, 'filename', 5, range(5))
        
    def test_cleans_before(self):
        self.fileslot.store_file('filename', 5, range(5))
        self.fileslot.cleaning_strategy.must_be_called_with(self.filedb, 'user_id', 23 - 5)
        
    def test_sets_attributes(self):
        self.fileslot.store_file('filename', 5, range(5))        
        assert self.fileslot.filename == 'filename'
        assert self.fileslot.size == 5

        
class TestFileslotStorage(FileSlotBase):

    def setup_method(self, method):
        self.setup_mocks()
        
    def test_store_file_in_filled_slot(self):
        # store first file
        self.fileslot.store_file('filename', 5, range(5))        
        # storing seconf file fails
        py.test.raises(self.filedb.UploadException, self.fileslot.store_file, 'filename', 5, range(5))
        
    def test_store_too_big_file(self):
        py.test.raises(self.filedb.UploadException, self.fileslot.store_file, 
                              'filename', self.user.max_file_size + 1, range(5))        
        self.filedb.store_error.must_be_called()                                             



class TestCleaningStrategy(object):
    def setup_method(self, method):
        self.deleted_ids = []
        self.fileslot_info = {
            1 : (None, None, None, 'error', '2023-05-01'),
            2 : (None, None, 2, None, '2023-05-02'),
            3 : (None, None, 3, None, '2023-05-03'),
            4 : (None, None, 4, None, '2023-05-04'),
            5 : (None, None, 5, None, '2023-05-05'),
        }
        self.filedb = testing.Object(
            get_fileslot_ids_for_user = testing.Method(returns=[1,2,3,4,5]),
            get_fileslot_info = self.fileslot_info.get,
            free_fileslot = self.deleted_ids.append,
        )
        self.fileslot = FileSlot(None, None, None, None, None, None, None, None)
        self.fileslot.cleaning_strategy(self.filedb, 'user_id', 9)
        
    def test_frees_correct_fileids(self):
        assert sorted(self.deleted_ids) == [2, 3]
        
