import pathfix

import py.test
import testhelper as testing

from skarabaeus_server.plugins.fsfiledb import FsFileDb
from skarabaeus_server.plugins.dictuserdb import DictUserDb

from skarabaeus_server.model import Model

class TestFsFileDb(object):

    def setup_method(self, method):        
        self.model = testing.Object(NotFound=Model.NotFound)
        
        self.filedb = FsFileDb(self.model, {})                            
        self.file_id = self.filedb.create_fileslot('username')
        
    def test_id_generation(self):
        """ a file should have a random id
        """
        assert 0 <= self.file_id
        
    def test_id_encoder(self):
        assert self.filedb._encode_id(2) == '2'
        assert self.filedb._encode_id(35) == 'Z'
        assert self.filedb._encode_id(36) == '10'
        
    
    
class TestDictUserDb(object):

    def setup_method(self, method):
        self.config = {
            'default_max_file_size' : 5,
            'default_max_total_size' : 17,
            'users' : {
                'discordia' : { 
                    'password' : 'passwd',
                }, 
                'anonymous' : {
                    'password' : None,
                },
                'mr_bigger_files' : {
                    'password' : 'passwd',
                    'max_file_size' : 23,
                },
                'mrs_bigger_total' : {
                   'password' : 'passwd',
                   'max_total_size' : 23,
                },
            },
        }        
        self.model = testing.Object(NotFound=Model.NotFound)                        
        self.userdb = DictUserDb(self.model, self.config)
        self.userdb.user_factory = testing.Method(returns='user')                                   
        
    def must_create_user(self, login, max_file_size, max_total_size):
        self.userdb.user_factory.must_be_called_with(self.model, login,
                                                  max_file_size, max_total_size)  
        
    def test_get_by_good_login(self):
        user = self.userdb.get_user_by_login('discordia')        
        assert user == 'user'
        self.must_create_user('discordia', 5, 17)
        
    def test_get_by_bad_login(self):
        py.test.raises(Model.NotFound, self.userdb.get_user_by_login, 'St. Claus')        
        
    def test_get_by_good_credentials(self):
        user = self.userdb.get_user_by_credentials('discordia', 'passwd')
        assert user == 'user'
        self.must_create_user('discordia', 5, 17)
                
    def test_get_by_bad_credentials(self):
        py.test.raises(Model.NotFound, self.userdb.get_user_by_credentials, 'St. Claus', 'xxx')
        py.test.raises(Model.NotFound, self.userdb.get_user_by_credentials, 'discordia', 'xxx')
                                        
    def test_get_anonymous_user(self):
        user = self.userdb.get_anonymous_user()
        assert user == 'user'
        self.must_create_user('anonymous', 5, 17)
                
    def test_bigger_file(self):
        self.userdb.get_user_by_login('mr_bigger_files')
        self.must_create_user('mr_bigger_files', 23, 17)

    def test_bigger_total(self):
        self.userdb.get_user_by_login('mrs_bigger_total')
        self.must_create_user('mrs_bigger_total', 5, 23)
    
