#!/usr/bin/python
"""
crkb_by.py - creation of the knowledge base from the statements generated by
the extraction script

Guide to execution:

python crkb_by.py [ACTION] [FOLDER1] [FOLDER2]

where ACTION is one of 'create' or 'compsim' and FOLDER1, FOLDER2 are
the input and output folders, respectively. The action 'create' creates the KB 
representation from the statements previously stored in FOLDER1, generating
the KB serialisation files in FOLDER2. If the action is 'compsim', the
script loads the KB serialisation from FOLDER1, computes the semantic 
similarities in it and stores the resulting knowledge base in FOLDER2.

Copyright (C) 2012 Vit Novacek (vit.novacek@deri.org), Digital Enterprise
Research Institute (DERI), National University of Ireland Galway (NUIG)
All rights reserved.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import os, sys, time
from skimmr_bm import util
from skimmr_bm.strg import *

if __name__ == "__main__":
  # reading the command line parameters
  action, in_path, out_path = 'create', os.getcwd(), os.getcwd()
  if len(sys.argv) > 1:
    action = sys.argv[1].lower()
  if len(sys.argv) > 2:
    in_path = os.path.abspath(sys.argv[2])
  if len(sys.argv) > 3:
    out_path = os.path.abspath(sys.argv[3])
  # setting the paths automatically if not specified
  if len(sys.argv) <= 2:
    if action == 'create':
      in_path = os.path.join(os.getcwd(),'text')
      out_path = os.path.join(os.getcwd(),'data','stre')
    elif action == 'compsim':
      in_path = os.path.join(os.getcwd(),'data','stre')
      out_path = os.path.join(os.getcwd(),'data','stre')
  if action == 'create':
    # creating a new store
    store = MemStore()
    print 'Loading the statements from:', in_path
    start = time.time()
    store.incorporate(in_path)
    end = time.time()
    print '...finished in %s seconds' % (str(end-start),)
    print 'Computing the corpus'
    start = time.time()
    store.computeCorpus()
    end = time.time()
    print '...finished in %s seconds' % (str(end-start),)
    print 'Normalising the corpus'
    start = time.time()
    store.normaliseCorpus()
    end = time.time()
    print '...finished in %s seconds' % (str(end-start),)
    print 'Exporting the store to:', out_path
    start = time.time()
    store.exp(out_path)
    end = time.time()
    print '...finished in %s seconds' % (str(end-start),)
  elif action == 'compsim':
    # computing the similarities in an existing store
    # maximum number of similar items
    SIM_LIM = 10
    store = MemStore(trace=True)
    print '*** Loading the store from:', in_path
    start = time.time()
    store.imp(in_path)
    end = time.time()
    print '...finished in %s seconds' % (str(end-start),)
    orig_size = len(store.corpus)
    print '  ... size as loaded:', len(store.corpus)
    # updating the lexicon with the similarity relation name
    store.lexicon.update(util.SIMR_RELNAME)
    rel_id = store.lexicon[util.SIMR_RELNAME]
    print '*** Computing the LAxLIRA perspective'
    start = time.time()
    store.computePerspective('LAxLIRA')
    end = time.time()
    print '...finished in %s seconds' % (str(end-start),)
    print '*** Initialising the analyser'
    start = time.time()
    analyser = Analyser(store,'LAxLIRA',compute=False)
    end = time.time()
    print '...finished in %s seconds' % (str(end-start),)
    print '*** Computing the similar terms...'
    # processing all the terms in the lexicon, computing the similarities
    #term_ids = store.lexicon.lex2int.values()
    # processing only average and higher frequencies
    start = time.time()
    term_ids = store.lexicon.sorted(limit=0,\
      ignored=['.*_[0-9]+$',util.COOC_RELNAME,util.SIMR_RELNAME])
    sim_dict = {}
    i = 0
    for t1 in term_ids:
      i += 1
      print '  ...', i, 'out of', len(term_ids)
      # getting a fresh similarity statement provenance index
      sims2src = {}
      # computing the similarities for the given term
      for t2, s in analyser.similarTo(t1,top=SIM_LIM,sims2src=sims2src):
        if not ((t1,rel_id,t2) in sim_dict or (t2,rel_id,t1) in sim_dict):
          # adding if a symmetric one was not added before
          sim_dict[(t1,rel_id,t2)] = s
    # storing the computed values to the corpus
    for key, value in sim_dict.items():
      store.corpus[key] = value
    end = time.time()
    print '...finished in %s seconds' % (str(end-start),)
    print '*** Exporting the updated store to:', out_path
    print '  ... size as loaded                 :', orig_size
    print '  ... size with similarities computed:', len(store.corpus)
    start = time.time()
    store.exp(out_path)
    end = time.time()
    print '...finished in %s seconds' % (str(end-start),)
  else:
    print 'Unknown action, try again'
