#
# Copyright (c) 2011 Andrey Churin.  All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS-IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# Author: Andrey Churin <aachurin@gmail.com>
#

from colander import Invalid, Schema

__all__ = ('Form', 'bind_schema')

class Form(object):
    """
    `schema` : Colander schema (default to None)
    `defaults` : a dict of default values (default to None)
    
    Use bind_schema decorator to create a form with the bound schema.
    """
    def __init__(self, schema=None, defaults=None):
        if schema:
            self.schema = schema

        assert self.schema, 'Maybe use bind_schema?'

        self.validated = False
        self.params = defaults or dict()
        self.errors = None
        self.data = None

    def get(self, key):
        """
        Returns field raw value
        """    
        return self.params.get(key, u'')

    def validate(self, params):
        """
        Runs validation and returns validated field values
        
        `params` : dictinory like object to validate
        
        Returns validated fields
        """
        if self.validated:
            return self.data

        self.validated = True

        self.params.update(params)
        try:
            self.data = self.schema.deserialize(self.params)
        except Invalid, e:
            self.errors = get_errors_from_exc(e)

        return self.data

    def bind(self, obj, path=None, include=None, exclude=None, mapping=None):
        """
        Binds validated field values to an object instance.

        `path` : path to object data to bind
        `include` : list of included fields.
        `exclude` : list of excluded fields.
        `mapping` : fields mapping

        Returns the `obj` passed in.

        Calling bind() before running validation will result in a RuntimeError
        """
        if not self.validated or self.errors:
            raise RuntimeError, 'Form has not been validated or has errors.'

        data = self.data
        
        if path:
            if isinstance(path, basestring):
                path = path.split('.')
            for i in path:
                data = data[i]

        mapping = mapping or dict()

        if include:
            items = [(k, data[mapping.get(k, k)]) for k in include]
        elif exclude:
            items = [(k, data[mapping.get(k, k)]) for k in data if k not in exclude]
        else:
            items = [(k, data[mapping.get(k, k)]) for k in data]

        for k, v in items:
            setattr(obj, k, v)

        return obj
    
    def has_errors(self, path):
        """
        Checks if there are errors for the given path
        
        Calling has_errors() before running validation will result in a RuntimeError
        """
        if not self.validated:
            raise RuntimeError, 'Form has not been validated.'
        if not self.errors:
            return False
        if hasattr(path, '__iter__'):
            path = '.'.join(map(str, path))
        return path in self.errors
        
    def get_errors(self, path):
        """
        Returns all errors for the given path or None
        
        Calling get_errors() before running validation will result in a RuntimeError
        """    
        if not self.validated:
            raise RuntimeError, 'Form has not been validated.'        
        if not self.errors:
            return None
        if hasattr(path, '__iter__'):
            path = '.'.join(map(str, path))
        return self.errors.get(path)
        
    def get_first_error(self, path):
        """
        Returns first error for the given path or None
        
        Calling get_first_error() before running validation will result in a RuntimeError
        """
        errors = self.get_errors(path)
        if errors:
            return errors[0]

    def add_errors(self, path, value):
        """
        Adds error(s) for the specified path
        
        Calling add_errors() before running validation will result in a RuntimeError
        """
        if not self.validated:
            raise RuntimeError, 'Form has not been validated.'
        if not self.errors:
            self.errors = dict()

        if hasattr(path, '__iter__'):
            path = '.'.join(map(str, path))

        errors = self.errors.setdefault(path, [])
        if hasattr(value, '__iter__'):
            errors.extend(value)
        else:
            errors.append(value)

def get_errors_from_exc(e):
    errors = dict()
    for path in e.paths():
        keyparts = []
        msgs = []
        for exc in path:
            keyname = exc._keyname()
            if keyname:
                keyparts.append(keyname)
            if exc.msg:
                msgs.extend(exc.messages())
        
        imsg = []
        for s in msgs:
            if hasattr(s, 'interpolate'):
                imsg.append(s.interpolate())
            else:
                imsg.append(s)
        errors['.'.join(keyparts)] = imsg
    return errors

def bind_schema(schema):
    """
    Creates form from schema
    """    
    if not issubclass(schema, Schema):
        raise TypeError('Schema expected')
    return type(schema.__name__, (Form,), dict(schema=schema()))

