# -*- coding: utf-8 -*-
"""
    soil.serving
    ~~~~~~~~~~~~

    Implements the Werkzeug application used to serve soil projects.

    :copyright: (c) 2012 by Natan Lao.
    :license: BSD, see LICENSE for more details.
"""
import os
import re

from soil.core import Soil

from werkzeug.serving import run_simple
from werkzeug.wrappers import Request, Response
from werkzeug.routing import Map, Rule
from werkzeug.exceptions import HTTPException


class SoilServer(Soil):
    """The Werkzeug application used to server soil projects.
    :param projdir: the directory of the soil project
    """
    def __init__(self, projdir):
        """:param projdir: the directory of the soil project"""
        super(SoilServer, self).__init__(projdir)

        rules = []

        for template in self.jinja_env.list_templates():
            head, tail = os.path.split(template)
            if re.match(self.config['templates_ignore'], tail):
                continue
            if os.path.basename(template) == 'index.html':
                path = '/' + os.path.dirname(template).rstrip('/') + '/'
                rules.append(Rule(path, endpoint=template))

            # Makes example.com/ = example.com/index.html
            # This doesn't work with the static folder because that's handled
            # through werkzeug.wsgi.SharedDataMiddleware
            rules.append(Rule('/' + template, endpoint=template))

        self.url_map = Map(rules)
        self.exec_extensions('pre')

    def render(self, template_name, **context):
        """Wrapper for `render_template()`- returns rendered template wrapped
        in a `Response` object. Accepts the same arguments as
        `render_template()`
        """
        return Response(self.render_template(template_name, **context),
                        mimetype='text/html')

    def error(self, code):
        """Returns a template corresponding to an error code. If no
        corresponding template can be found, `error()` falls back
        to the error 500 template.
        :param code: the HTTP error code
        """
        if int(code) in self.config['error_pages']:
            template_path = self.config['error_pages'][code]
        else:
            # Fall back to error 500
            template_path = self.config['error_pages'][500]

        response = self.render(template_path)
        response.status_code = code

        return response

    def dispatch_request(self, request):
        """Dispatches requests by matching the requested URL to a
        template in the 'templates' directory specified in soilcfg.py.
        :param request: Werkzeug request object
        """
        adapter = self.url_map.bind_to_environ(request.environ)

        try:
            endpoint, values = adapter.match()
            return self.render(endpoint)
        except HTTPException, e:
            return self.error(e.code)

    def wsgi_app(self, environ, start_response):
        request = Request(environ)
        response = self.dispatch_request(request)
        return response(environ, start_response)

    def __call__(self, environ, start_response):
        return self.wsgi_app(environ, start_response)


def create_app(projdir):
    """Creates an instance of SoilServer.
    :param projdir: the directory of the soil project
    """
    app = SoilServer(projdir)
    return app


def serve(config, projdir, **options):
    """Runs `SoilServer`. Any arguments passed after `config` and `projdir`
    will be passed to `werkzeug.serving.run_simple`.
    :param config: a dictionary containing the parsed contents of soilcfg.py,
                   usually the output of Soil.load_config()
    :param projdir: the directory in which the soil project is located
    """
    app = create_app(projdir)

    params = {
        'hostname': config['listen_addr'][0],
        'port': config['listen_addr'][1],
        'application': app,
        'use_reloader': True,
        'use_debugger': True,
        'use_evalex': True,
        'extra_files': [config['__file__']],
        'threaded': True,
        'static_files': {'/static': config['directories']['static'][1]},
    }

    params.update(options)
    run_simple(**params)
