#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
    soil.shovel
    ~~~~~~~~~~~

    Implements the `shovel` CLI tool frontend - parses arguments and
    handles creation of new projects.

    :copyright: (c) 2012 by Natan Lao.
    :license: BSD, see LICENSE for more details.
"""
from __future__ import print_function, with_statement

import os
import sys

from shutil import copyfile

import soil
from soil.util import status, skeleton_path
from docopt import docopt
from termcolor import colored

cli_doc = """
Usage:
  shovel new <directory>
  shovel build [-d DIR] [-o DIR]
  shovel run [-d DIR]
  shovel -h | --help
  shovel --version

Commands:
  new         Create a new soil project.
  build       Generates static html from the project.
  run         Runs the project locally on a server.

Arguments:
  -h, --help  Display this help text.
  -d DIR      Run the command on a project not in the
              current working directory. Will be
              ignored when used with the `new` command.
  --version   Show version.

shovel new:
  directory   The directory in which to create the new
              soil project. Only usable with the `new`
              command.

shovel build:
  -o DIR      Store the generated pages in a different
              directory than build/
"""


def new(projdir):
    """Creates a new soil project by copying the contents of soil.skeleton
    to `projdir`.
    :param projdir: the directory in which to create the new project
    """

    src_dir = skeleton_path
    dst_dir = os.path.realpath(projdir)

    for folder, folders, files in os.walk(src_dir):
        #: Path of the to-be-copied folder for printing
        src_folder_fancy = os.path.relpath(folder, src_dir).rstrip(os.path.sep)
        src_folder_fancy += os.path.sep
        #: Path of the to-be-copied folder in src_dir
        src_folder = os.path.abspath(os.path.join(dst_dir, src_folder_fancy))

        # If a required folder doesn't exist, create it.
        if not os.path.isdir(src_folder):
            os.makedirs(src_folder)
            print(status['create'], src_folder_fancy)
        else:
            print(status['exists'], src_folder_fancy)

        # Copy each file to dst_dir
        for fname in files:
            if fname.endswith(('.pyc', '.pyo', '.DS_Store')):
                continue

            # Absolute path of the source file.
            sfilepath = os.path.realpath(os.path.join(folder, fname))
            # Absolute path of the destination file.
            dfilepath = os.path.join(dst_dir,
                                     os.path.relpath(folder, src_dir), fname)
            dfilepath = os.path.realpath(dfilepath)
            # Relative path of the destination file.
            rfilepath = os.path.relpath(dfilepath)

            # If the file does not exist in the destination dir
            if not os.path.isfile(dfilepath):
                copyfile(sfilepath, dfilepath)
                print(status['create'], rfilepath)
                continue
            # If the file exists in the destination dir...
            elif os.path.isfile(dfilepath):
                # ... and is the same as the file in the source dir ...
                if os.path.getsize(sfilepath) == os.path.getsize(dfilepath):
                    with open(sfilepath, 'r') as sfile:
                        with open(dfilepath, 'r') as dfile:
                            if sfile.read() == dfile.read():
                                # ... then don't copy it.
                                print(status['exists'], rfilepath)
                                continue
                # ... and the dest file isn't the same as the source file...
                else:
                    # ... then ask what to do.
                    print(status['conflict'], rfilepath)
                    choice = None

                    while not choice:
                        choice = raw_input('Replace %s? [Yn] ' %
                                       colored(dfilepath, attrs=['underline']))
                        choice = choice.lower()

                        if choice == 'y' or choice == '':
                            print(status['overwrite'], rfilepath)
                            copyfile(sfilepath, dfilepath)
                            break
                        elif choice == 'n':
                            print(status['skip'], rfilepath)
                            break
                        else:
                            choice = None

    print("You're all set! `cd` into your project folder and do "
          '`shovel run` to run your project.')


def main(argv):
    """`shovel` main function. Ensures that there is an accessible project
    when required and routes arguments.

    :param argv: `sys.argv`, or a list of arguments in a similar format
    """
    args = docopt(cli_doc, argv=argv[1:], help=True, version=soil.__version__)

    # Find the correct project directory
    if not argv[1] == 'new':
        proj = args['-d'] or os.getcwd()
        # Also, if the command is not `new`, that means we're operating on an
        # exisiting project.
        assert os.path.exists(os.path.join(proj, 'soilcfg.py')), \
               "Didn't find the project's `soilcfg.py`."

        app = soil.Soil(os.path.realpath(proj))

    if argv[1] == 'new':
        new(sys.argv[2])
    elif argv[1] == 'build':
        app.build()
    elif argv[1] == 'run':
        app.run()

    exit()


def run_main():
    """Runs the `main()` function, setting the `argv` parameter to sys.argv."""
    main(sys.argv)

if __name__ == '__main__':
    main(sys.argv)
