# -*- coding: utf-8 -*-
"""
    soil.test.fixtures
    ~~~~~~~~~~~~~~~~~~

    Contains py.test fixtures for use in tests.

    :copyright: (c) 2012 by Natan Lao.
    :license: BSD, see LICENSE for more details.
"""
import os
import pytest

from shutil import rmtree

from soil import Soil, shovel
from soil.util import skeleton_path


test_proj_dir = os.path.join(os.path.dirname(__file__), 'test_project')
test_proj_dir = os.path.realpath(test_proj_dir)
global test_project_dir


@pytest.fixture
def test_project(request):
    """Fixture that allows access to the test project
    (`soil.test.test_project`). Wraps :data:`test_project_dir` and
    implements a teardown function that removes the 'build' directory.
    """
    build_dir = os.path.join(test_proj_dir, 'build')

    if not os.path.exists(build_dir):
        os.makedirs(build_dir)

    def teardown():
        if os.path.exists(build_dir):
            rmtree(build_dir)

    request.addfinalizer(teardown)
    return test_proj_dir


@pytest.fixture(scope='session')
def skeleton():
    """Returns the path to the application skeleton :data:`soil.skeleton`."""
    return skeleton_path


@pytest.fixture
def project(tmpdir):
    """Gives access to a new, blank project. Unlike :func:`test_project()`,
    which returns the test project (`soil.test.test_project`) which has been
    customized and tweaked for testing, :func:`project()` returns a new,
    untouched project.
    """
    tmpdir = str(tmpdir)
    shovel.new(tmpdir)
    return tmpdir


@pytest.fixture
def soil_app(test_project):
    """Returns an instance of :class:`soil.Soil` set to the test project
    returned by :func:`test_project()`
    """
    app = Soil(test_project)
    return app


@pytest.fixture
def new_soil_app(tmpdir):
    """Returns an instance of :class:`soil.Soil` set to a new, blank project.
    Like :func:`soil_app()`, except with a new project.
    """
    shovel.new(str(tmpdir))
    app = Soil(str(tmpdir))
    return app
