# -*- coding: utf-8 -*-
"""
    soil.test.test_shovel
    ~~~~~~~~~~~~~~~~~~~~~

    Contains tests for the `shovel` utility, namely-
    * project creation
    * argument parsing

    :copyright: (c) 2012 by Natan Lao.
    :license: BSD, see LICENSE for more details.
"""
import pytest
from soil import shovel
from soil.test.fixtures import project, skeleton, test_project
import os


class TestProjectCreation:
    """Contains test cases for testing how `shovel` creates projects."""

    def test_dest_hierarchy(self, project, skeleton):
        """Checks if the newly-copied files have the same structure as the
        skeleton.
        """
        os.chdir(project)
        src_folder = []
        dst_folder = []

        for folder, folders, files in os.walk(skeleton):
            for fname in files:
                path = os.path.relpath(os.path.join(folder, fname), skeleton)
                src_folder.append(path)

        for folder, folders, files in os.walk(project):
            for fname in files:
                path = os.path.relpath(os.path.join(folder, fname), project)
                dst_folder.append(path)

        for fname in src_folder:
            if fname.endswith(('.pyc', '.pyo', '.DS_Store')):
                continue

            assert fname in dst_folder

    def test_dest_integrity(self, project, skeleton):
        """Checks if the new soil project contains the right files."""
        os.chdir(project)
        src_folder = {}
        dst_folder = {}

        for folder, folders, files in os.walk(skeleton):
            for fname in files:
                abspath = os.path.realpath(os.path.join(folder, fname))
                relpath = os.path.relpath(abspath, skeleton)
                src_folder[relpath] = abspath

        for folder, folders, files in os.walk(project):
            for fname in files:
                abspath = os.path.realpath(os.path.join(folder, fname))
                relpath = os.path.relpath(abspath, project)
                dst_folder[relpath] = abspath

        for sfname, abs_sfname in src_folder.iteritems():
            if sfname.endswith(('.pyc', '.pyo', '.DS_Store')):
                continue

            with open(abs_sfname, 'r') as sfile:
                with open(dst_folder[sfname], 'r') as dfile:
                    assert sfile.read() == dfile.read()


class TestArgumentParsing:
    """Contains test cases for verifying `shovel`'s behavior when parsing
    arguments.
    """

    def test_bad_args(self, test_project):
        """Tests for incorrect combinations of commands and arguments."""
        os.chdir(test_project)

        argv = ['shovel.py', 'run', '-o', 'sdfsdf']
        with pytest.raises(SystemExit) as e:
            shovel.main(argv)
            assert e.code == 1

    def test_project_detection(self, tmpdir):
        """Tests if the lack of a soil project can be detected."""
        argv = ['shovel.py', 'run']
        tmpdir.chdir()

        with pytest.raises(AssertionError) as e:
            shovel.main(argv)
            assert e.message == "Didn't find the project's `soilcfg.py`."
