#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
    soil.core
    ~~~~~~~~~

    Contains the main Soil class.

    :copyright: (c) 2012 by Natan Lao.
    :license: BSD, see LICENSE for more details.
"""
from __future__ import print_function

import imp
import os
import re

from soil.util import status, getattr_nested

from jinja2 import Environment, FileSystemLoader
from termcolor import colored


class Soil(object):
    """The main Soil class. Implements configuration loading, template
    rendering, project building, and project serving.

    :param projdir: directory of the soil project
    """
    def __init__(self, projdir):
        self.projdir = projdir
        self.config = self.load_config()
        self.jinja_env = self.make_jinja_env(self.config['jinja_opts'])
        self.jinja_context = {}
        self.ext_entry_points = {
            'pre': [],
            'post': []
        }
        self.extensions = {}
        self.watched_files = [self.config['__file__']]

        # Get the actual extension class from each extension
        for extname in self.config['extensions']:
            ext = getattr_nested(extname)
            self.extensions[extname] = []

            for name in dir(ext):
                if not name.startswith('_') and name.endswith('Extension'):
                    self.extensions[ext.__name__].append(
                        getattr(ext, name)(self))

    def load_config(self):
        """Loads a soilcfg.py file and returns its contents in a dictionary.
        The returned dictionary will be identical to soilcfg.py except for
        the following additions:

        * directories specified in the 'directories' option will be tupled
          with their complementary absolute paths

        * a key named `__file__` will be added to the returned dictionary
          with the absolute path of the soilcfg.py being parsed.

        """
        cfgpath = os.path.realpath(os.path.join(self.projdir, 'soilcfg.py'))
        cfg = imp.load_source('soilcfg', cfgpath)

        cfg = {k: getattr(cfg, k) for k in dir(cfg) if not k.startswith('_')}

        # 'relative_path' > ('relative_path', 'absolute_path')
        for k, v in cfg['directories'].iteritems():
            abspath = os.path.realpath(os.path.join(self.projdir, v))
            cfg['directories'][k] = (v, abspath)

        cfg['__file__'] = cfgpath

        return cfg

    def make_jinja_env(self, options):
        """Creates a :class:`jinja2.Environment` object.

        :param options: dictionary of options to pass to the Jinja2 environment
                        object
        """
        template_dir = os.path.join(self.config['directories']['templates'][1])
        rv = Environment(loader=FileSystemLoader(template_dir,
                                                 encoding='utf-8'), **options)
        return rv

    def render_template(self, template_name, **context):
        """Renders a template from the template folder with the given
        context.

        :param template_name: name of the template to render relative to the
                              'templates' directory
        :param context: variables that should be available in the context of
                        the template
        """
        t = self.jinja_env.get_template(template_name)
        return t.stream(context, **self.jinja_context)

    def register_ext_entry_points(self, entry_points):
        """Registers an extension's entry point tasks in
        :data:`self.ext_entry_points`.

        :param entry_points: a dictionary containing the extension's task,
                             corresponding to entry point
        """
        for k, v in entry_points.iteritems():
            if k in ('pre', 'post'):
                self.ext_entry_points[k].append(v)

    def exec_extensions(self, entry_point):
        """Executes all extension tasks for a specified entry point.

        :param entry_point: the entry point to execute tasks for
        """
        extlist = self.ext_entry_points[entry_point]
        filter(lambda ext: extlist[extlist.index(ext)](), extlist)

    def build(self):
        """Builds the soil project and stores the output in the 'build'
        directory defined in soilcfg.py.
        """
        build_dir = self.config['directories']['build'][1]
        template_dir = self.config['directories']['templates'][1]

        if not os.path.exists(build_dir):
            os.makedirs(build_dir)

        print('Building project...')
        self.exec_extensions('pre')

        for template in self.jinja_env.list_templates():
            head, tail = os.path.split(template)
            head_path = os.path.join(build_dir, head)

            if re.match(self.config['templates_ignore'], tail):
                continue

            if head and not os.path.exists(head_path):
                os.makedirs(head_path)

            t_path = os.path.join(build_dir, template)
            self.render_template(template).dump(t_path, encoding='utf-8')

            print(status['success'], template)

        self.exec_extensions('post')
        print('Compiled pages stored in %s' %
              colored(build_dir, attrs=['underline']))

    def run(self, **params):
        """Runs the soil project on the address specified in soilcfg.py with
        the given parameters.

        :param params: options to pass to :func:`werkzeug.serving.run_simple`.
        """
        import soil.serving
        params.update(extra_files=self.watched_files)
        soil.serving.serve(self.config, self.projdir, **params)
