__author__ = "ardevelop"

import urllib
import urlparse
import httplib
import socket
try:
    import ujson as json
except ImportError:
    import json

REQUEST_HEADERS = {'Content-Type': 'application/json'}


class HTTPError(Exception):
    def __init__(self, code, message):
        self.code = code
        self.message = message

    def __str__(self):
        return "\nResponse code: %s\nResponse: %s" % (self.code, self.message)


class ConnectionError(Exception):
    def __init__(self, message):
        self.message = message

    def __str__(self):
        return "\nOperationError: %s" % self.message


class SolrConnection(object):
    def __init__(self, url, reconnect=1):
        self.url = url.strip('/')
        self.update_url = self.url + '/update'
        self.update_url_with_commit = self.url + '/update?commit=true'
        self.select_url = self.url + '/select'
        self.reconnect = reconnect
        self.scheme, self.host, self.path = urlparse.urlparse(url, 'http')[:3]
        self.conn = httplib.HTTPConnection(self.host, timeout=None)

    def _reconnect(self):
        self.conn.close()
        self.conn.connect()

    def request(self, verb, url, obj=None):
        attempt = 0
        while attempt < self.reconnect:
            attempt += 1
            try:
                self.conn.request(verb, url, json.dumps(obj) if obj else None, REQUEST_HEADERS)
                response = self.conn.getresponse()
                response_body = response.read()
                if 200 == response.status:
                    return json.loads(response_body)
                else:
                    raise HTTPError(response.status, response_body)
            except (socket.error, httplib.ImproperConnectionState, httplib.BadStatusLine):
                self._reconnect()

        return ConnectionError("Cannot establish connection with server" if self.reconnect else "Connection closed")

    def select(self, query, **kwargs):
        kwargs['q'] = query
        kwargs['wt'] = 'json'
        return self.request('GET', self.select_url + '?' + urllib.urlencode(kwargs, 1))

    def add(self, **fields):
        return self.request('POST', self.update_url, {"add": {"doc": fields}})

    def update(self, doc_id, commit=False, **fields):
        data = {'id': doc_id}
        for field, value in fields.items():
            data[field] = {'set': value}
        return self.request('POST', self.update_url_with_commit if commit else self.update_url, [data])

    def delete_doc(self, doc_id):
        return self.request('POST', self.update_url, {"delete": {"id": doc_id}})

    def delete_query(self, query):
        return self.request('POST', self.update_url, {"delete": {"query": query}})


