"""This is a python module to help in delivery of mail to multiple
mailboxes, similar to procmail and the Perl module Mail::Sort.  There is
necessarily some overlap with the email package bundled with Python but
this one is optimized for fast matching on headers and fast delivery,
and treats the body of the mail as just a string, i. e. there is no
specific MIME support.  This is a free package, please see the file
LICENSE in this directory for the terms of use.  Documentation is
available via pydoc.

Apart from the functions and classes, the module exports the variable
log_formatter - a pre-made formatter object for logging.FileHandler.SetFormatter

Here's an example delivery script (some names altered to protect the guilty):
> #! /usr/bin/python
> 
> import sortmail
> from sortmail import msg, destination
> import sys
> import logging
> 
> md = '/home/myself/Mail'
> dbfile = md + '/msgid.DB_File'
> spool = md + '/spool'
> handler = logging.FileHandler (md + '/procmail.log')
> handler.setFormatter (sortmail.log_formatter)
> logger = logging.getLogger ('sortmail')
> logger.addHandler (handler)
> logger.setLevel (logging.DEBUG)
> 
> sort = msg.Msg (source = sys.stdin)
> 
> # delete useless relay spam checking headers
> useless_spam_headers = ['x-spam', 'x-attachments', 'x-miltered', 'x-j-chkmail-score', 'x-nai-spam-rules']
> map (lambda h: sort.delete_header_tag_all (r'^\s*' + h), useless_spam_headers)
> 
> # Administrative automated messages
> if (sort.header_match ('From', '.*Mailer-Daemon@unicorn')
>     or (sort.header_start ('From', 'root')
>         and sort.header_start ('To', 'root')
>         and sort.header_start ('Subject', 'apt-listchanges'))
>     or (sort.header_start('From', r'root@unicorn\.ahiker\.homeip\.net \(Cron Daemon\)'))
>     or (sort.header_start('From', 'Anacron')
>         and sort.header_start('To', '(root|myself)@unicorn')
>         and sort.header_start('Subject', 'Anacron job'))
>     or (sort.header_start('From', 'changetrack@unicorn')
>         and sort.header_start('To', 'root@(unicorn|localhost)'))
>     or (sort.header_start('From', 'root')
>         and sort.header_start('To', 'root')
>         and sort.header_start('Subject', 'Debian Package Updates'))
>     or (sort.header_start('From', 'root')
>         and sort.header_start('Subject', 'Daily exim4 reject'))
>     or (sort.header_start('From', 'root')
>         and sort.header_start('To', 'root')
>         and sort.header_start('Subject', 'fwlogwatch'))
>     or (sort.header_start('From', 'root')
>         and sort.header_start('To', 'root')
>         and sort.header_start('Subject', 'List of changed'))):
>     sort.deliver ([destination.MBox (spool + '/mail.local.spool')], dbfile)
>     sys.exit (0)
> 
> sort.filtermsg (['spamoracle', 'mark'])
> if sort.header_start ('x-spam', 'yes'):
>     sort.deliver ([destination.MBox (spool + '/junk.spool')], dbfile)
>     sys.exit (0)
> 
> # Amazon & similar
> if sort.destination_address ('ard@my.domain.org'):
>     sort.deliver ([destination.MBox (spool + '/mail.orders.spool')], dbfile)
>     sys.exit (0)
> 
> mbox_dests = set ()
> 
> # Real mail
> ngs = ['comp.lang.functional', 'comp.lang.haskell', 'comp.text.sgml',
>        'comp.theory', 'comp.unix.programmer', 'sci.math.research']
> 
> for ng in ngs:
>     if sort.newsgroup_match (ng):
>         mbox_dests.add (ng)
> 
> lists = [('buug@weak.org', 'mail.buug'), ('clpm@lists.eyrie.org', 'mail.clpm'),
>          ('compilers@iecc.com', 'mail.compiler'), ('docbook-apps@lists.oasis-open.org', 'mail.docbook'),
>          ('dssslist@lists.mulberrytech.com', 'mail.dsssl'), ('help-gnu-emacs@gnu.org', 'mail.emacs-help'),
>          ('gnu-emacs-sources@gnu.org', 'mail.emacs-sources'), ('frogs@prooftheory.org', 'mail.frogs'),
>          ('forum@mail.gap-system.org', 'mail.gap'), ('glasgow-haskell-users@haskell.org', 'mail.ghc'),
>          ('gmp-discuss@swox.com', 'mail.gmp'), ('info-gnus-english@gnu.org', 'mail.gnus'),
>          ('group-pub-forum@lists.maths.bath.ac.uk', 'mail.group-pub'), ('haskell@haskell.org', 'mail.haskell'),
>          ('haskell-cafe@haskell.org', 'mail.haskell'), ('haskell-prime@haskell.org', 'mail.haskell'),
>          ('hol-info@lists.sourceforge.net', 'mail.hol-info'), ('hugs-users@haskell.org', 'mail.hugs'),
>          ('cl-isabelle-users@lists.cam.ac.uk', 'mail.isabelle'), ('mlton-user@mlton.org', 'mail.mlton'),
>          ('caml-list@yquem.inria.fr', 'mail.ocaml'), ('polyml@inf.ed.ac.uk', 'mail.polyml'),
>          ('list@prooftheory.org', 'mail.prooftheory'), ('python-announce-list@python.org', 'mail.python-announce'),
>          ('sawfish-list@gnome.org', 'mail.sawfish'), ('types-list@lists.seas.upenn.edu', 'mail.types'),
>          ('help-gnu-utils@gnu.org', 'mail.utils-help')]
> 
> for (lst_id, lst_mbox) in lists:
>     if sort.list_match (lst_id):
>         mbox_dests.add (lst_mbox)
> 
> if (sort.header_match ('list-archive', '.*coq-club')
>     or sort.destination_match ('.*coq-club@[^, ]inria')):
>     mbox_dests.add ('mail.coq')
> 
> if (sort.destination_match (r'.*[0-9]+@bugs\.debian\.org')
>     or sort.sender_match ('.*Debian BTS')):
>     mbox_dests.add ('mail.debian-bugs')
> 
> if (sort.sender_match ('(?:.*[^-a-z0-9_.])xml-l@')
>     or sort.destination_address ('xml-l@listserv.heanet.ie')):
>     mbox_dests.add ('mail.xml')
> 
> if len (mbox_dests) != 0:
>     sort.deliver (map (lambda b: destination.MBox (spool + '/' + b + '.spool'), mbox_dests), dbfile)
> elif sort.header_start ('from', 'Myself'):
>     sort.deliver ([destination.MBox (spool + '/mail.misc.spool')], dbfile)
> else:
>     sort.deliver ([destination.MBox (spool + '/mail.other')], dbfile)
> 
> sys.exit (0)
"""

# Copyright (C) 2007, 2010 Ian Zimmerman <itz@buug.org>

# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the conditions spelled out in
# the file LICENSE are met.

from __future__ import with_statement

__author__ = 'Ian Zimmerman <itz@buug.org>'
__date__ = '2010-08-02'

__all__ = ['msg', 'destination', 'lock', 'rfc2047', 'log_formatter',
           'restoring_loglevel', 'with_log_level']

import logging
from contextlib import contextmanager

log_formatter = logging.Formatter ('%(asctime)s [%(process)d] %(message)s')

@contextmanager
def restoring_loglevel(level):
    logger = logging.getLogger ('sortmail')
    old_level = logger.getEffectiveLevel ()
    logger.setLevel (level)
    try:
        yield
    finally:
        logger.setLevel(old_level)

def with_log_level (level, fun):
    """Call the function like object fun with logging level locally set to level.
    The previous effective level is restored when with_log_level returns or
    propagates an exception.
    """
    with restoring_loglevel(level):
        return fun()
